import { generatePageMetadata } from "@/lib/seo/metadata"
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card"
import {
  Users,
  TrendingUp,
  ArrowUpRight,
  ArrowDownRight,
  ChevronRight,
  Hotel,
  Car,
  Package,
  ClipboardList,
  DollarSign,
  Wallet,
  UsersRound,
  Briefcase,
  MapPin,
  Map,
  Building2,
  CalendarCheck,
  CarFront,
  Route,
  PackageCheck,
  Sparkles,
  Settings2,
} from "lucide-react"
import type { Metadata } from "next"
import { cn } from "@/lib/utils"
import Link from "next/link"

export const metadata: Metadata = generatePageMetadata({
  title: "Admin Dashboard",
  description: "Teertham Platform administration dashboard",
  path: "/admin/dashboard",
  noIndex: true,
})

const stats = [
  {
    name: "Total Users",
    value: "2,543",
    change: "+12.5%",
    trend: "up",
    icon: Users,
    color: "from-violet-500 to-purple-600",
    description: "from last month",
  },
  {
    name: "Active Agents",
    value: "456",
    change: "+8.2%",
    trend: "up",
    icon: Briefcase,
    color: "from-blue-500 to-cyan-500",
    description: "from last month",
  },
  {
    name: "Hotel Bookings",
    value: "1,234",
    change: "+23.1%",
    trend: "up",
    icon: Hotel,
    color: "from-emerald-500 to-teal-500",
    description: "from last month",
  },
  {
    name: "Vehicle Bookings",
    value: "892",
    change: "+15.7%",
    trend: "up",
    icon: Car,
    color: "from-amber-500 to-orange-500",
    description: "from last month",
  },
  {
    name: "Package Bookings",
    value: "567",
    change: "+18.9%",
    trend: "up",
    icon: Package,
    color: "from-pink-500 to-rose-500",
    description: "from last month",
  },
  {
    name: "Agent Earnings",
    value: "₹4.2L",
    change: "+21.3%",
    trend: "up",
    icon: DollarSign,
    color: "from-green-500 to-emerald-600",
    description: "from last month",
  },
  {
    name: "Pending Withdrawals",
    value: "₹85K",
    change: "-5.2%",
    trend: "down",
    icon: Wallet,
    color: "from-red-500 to-pink-600",
    description: "from last month",
  },
  {
    name: "Growth Rate",
    value: "15.3%",
    change: "-2.4%",
    trend: "down",
    icon: TrendingUp,
    color: "from-indigo-500 to-purple-600",
    description: "from last month",
  },
]

const userStats = [
  { label: "Total Customers", value: "2,087", icon: UsersRound, href: "/admin/customers" },
  { label: "Total Agents", value: "456", icon: Briefcase, href: "/admin/agents" },
]

const locationStats = [
  { label: "States Covered", value: "28", icon: Map, href: "/admin/states" },
  { label: "Cities Available", value: "156", icon: MapPin, href: "/admin/cities" },
]

const hotelStats = [
  { label: "Hotel Partners", value: "342", icon: Building2, href: "/admin/hotel-partners" },
  { label: "Total Hotels", value: "1,245", icon: Hotel, href: "/admin/all-added-hotels" },
  { label: "Hotel Bookings", value: "1,234", icon: CalendarCheck, href: "/admin/hotels-booking" },
]

const vehicleStats = [
  { label: "Vehicle Listings", value: "486", icon: CarFront, href: "/admin/vehicle-listing" },
  { label: "Active Routes", value: "89", icon: Route, href: "/admin/routes-listing" },
  { label: "Vehicle Bookings", value: "892", icon: CalendarCheck, href: "/admin/vehicle-bookings" },
]

const packageStats = [
  { label: "Fix Packages", value: "45", icon: PackageCheck, href: "/admin/fix-packages" },
  { label: "Spiritual Packages", value: "67", icon: Sparkles, href: "/admin/spiritual-packages" },
  { label: "Custom Packages", value: "123", icon: Settings2, href: "/admin/custom-packages" },
]

const agentFinancials = [
  { label: "Agent Bookings", value: "2,145", icon: ClipboardList, href: "/admin/agent-bookings" },
  { label: "Total Earnings", value: "₹4.2L", icon: DollarSign, href: "/admin/agent-earnings" },
  { label: "Withdrawals", value: "₹85K", icon: Wallet, href: "/admin/agent-withdraw" },
]

const recentActivity = [
  { message: "New user registered", time: "5 minutes ago", type: "success" },
  { message: "Agent application submitted", time: "15 minutes ago", type: "info" },
  { message: "Hotel booking confirmed", time: "32 minutes ago", type: "success" },
  { message: "Vehicle booking cancelled", time: "1 hour ago", type: "warning" },
  { message: "Package booking completed", time: "2 hours ago", type: "success" },
  { message: "Customer blocked by admin", time: "3 hours ago", type: "warning" },
]

const quickActions = [
  { title: "User Management", description: "Manage customer and agent accounts", href: "/admin/customers" },
  { title: "Hotel Partners", description: "Review and manage hotel partnerships", href: "/admin/hotel-partners" },
  { title: "Vehicle Listings", description: "Manage vehicle inventory and routes", href: "/admin/vehicle-listing" },
  { title: "Package Control", description: "Configure and manage tour packages", href: "/admin/fix-packages" },
]

function StatSection({
  title,
  stats,
  gradient,
}: {
  title: string
  stats: { label: string; value: string; icon: any; href: string }[]
  gradient: string
}) {
  return (
    <Card className={cn("admin-card rounded-xl border-0 overflow-hidden relative", gradient)}>
      <div className="absolute inset-0 bg-gradient-to-br opacity-5" />
      <CardHeader className="relative">
        <CardTitle className="text-base">{title}</CardTitle>
      </CardHeader>
      <CardContent className="relative">
        <div className="grid grid-cols-1 gap-3">
          {stats.map((stat) => {
            const Icon = stat.icon
            return (
              <Link
                key={stat.label}
                href={stat.href}
                className="flex items-center justify-between p-3 rounded-lg bg-card/80 backdrop-blur-sm hover:bg-card transition-all hover:scale-[1.02] group"
              >
                <div className="flex items-center gap-3">
                  <div className="p-2 rounded-lg bg-muted/50 group-hover:bg-primary/10 transition-colors">
                    <Icon className="h-4 w-4 text-muted-foreground group-hover:text-primary transition-colors" />
                  </div>
                  <span className="text-sm font-medium">{stat.label}</span>
                </div>
                <span className="text-lg font-bold">{stat.value}</span>
              </Link>
            )
          })}
        </div>
      </CardContent>
    </Card>
  )
}

export default function AdminDashboardPage() {
  return (
    <div className="admin-section">
      <header className="admin-page-header">
        <h1 className="admin-page-title">Dashboard</h1>
        <p className="admin-page-description">Welcome back! Here's an overview of your platform.</p>
      </header>

      <section aria-label="Statistics overview">
        <div className="grid gap-4 sm:grid-cols-2 lg:grid-cols-4">
          {stats.map((stat) => {
            const Icon = stat.icon
            const TrendIcon = stat.trend === "up" ? ArrowUpRight : ArrowDownRight
            return (
              <Card key={stat.name} className="admin-card relative overflow-hidden rounded-xl border-0">
                <div className={cn("absolute inset-0 bg-gradient-to-br opacity-5", stat.color)} />
                <CardHeader className="flex flex-row items-center justify-between pb-2 relative">
                  <CardTitle className="text-sm font-medium text-muted-foreground">{stat.name}</CardTitle>
                  <div className={cn("rounded-xl bg-gradient-to-br p-2.5 shadow-lg", stat.color)}>
                    <Icon className="h-4 w-4 text-white" aria-hidden="true" />
                  </div>
                </CardHeader>
                <CardContent className="relative">
                  <div className="text-3xl font-bold">{stat.value}</div>
                  <div className="flex items-center gap-1.5 mt-2">
                    <div
                      className={cn(
                        "flex items-center gap-0.5 px-1.5 py-0.5 rounded-md text-xs font-semibold",
                        stat.trend === "up"
                          ? "bg-emerald-500/10 text-emerald-600"
                          : "bg-destructive/10 text-destructive",
                      )}
                    >
                      <TrendIcon className="h-3 w-3" aria-hidden="true" />
                      {stat.change}
                    </div>
                    <span className="text-xs text-muted-foreground">{stat.description}</span>
                  </div>
                </CardContent>
              </Card>
            )
          })}
        </div>
      </section>

      <section aria-label="Category overview" className="grid gap-6 md:grid-cols-2 lg:grid-cols-3">
        <StatSection title="User Management" stats={userStats} gradient="from-violet-500/10 to-purple-500/5" />
        <StatSection title="Location Coverage" stats={locationStats} gradient="from-blue-500/10 to-cyan-500/5" />
        <StatSection title="Hotel Management" stats={hotelStats} gradient="from-emerald-500/10 to-teal-500/5" />
        <StatSection title="Vehicle Management" stats={vehicleStats} gradient="from-amber-500/10 to-orange-500/5" />
        <StatSection title="Package Control" stats={packageStats} gradient="from-pink-500/10 to-rose-500/5" />
        <StatSection title="Agent Financials" stats={agentFinancials} gradient="from-green-500/10 to-emerald-500/5" />
      </section>

      <div className="grid gap-6 lg:grid-cols-2">
        <section aria-label="Quick actions">
          <h2 className="text-lg font-semibold mb-4">Quick Actions</h2>
          <div className="grid gap-3">
            {quickActions.map((action) => (
              <Link key={action.title} href={action.href}>
                <Card className="admin-card cursor-pointer transition-all hover:scale-[1.02] rounded-xl border-0 group">
                  <CardHeader className="pb-2 flex flex-row items-center justify-between">
                    <div>
                      <CardTitle className="text-base group-hover:text-primary transition-colors">
                        {action.title}
                      </CardTitle>
                      <CardDescription>{action.description}</CardDescription>
                    </div>
                    <ChevronRight className="h-5 w-5 text-muted-foreground group-hover:text-primary group-hover:translate-x-1 transition-all" />
                  </CardHeader>
                </Card>
              </Link>
            ))}
          </div>
        </section>

        <section aria-label="Recent activity">
          <h2 className="text-lg font-semibold mb-4">Recent Activity</h2>
          <Card className="admin-card rounded-xl border-0">
            <CardHeader className="pb-3">
              <CardTitle className="text-base">Activity Log</CardTitle>
              <CardDescription>Latest actions and events on your platform</CardDescription>
            </CardHeader>
            <CardContent>
              <div className="space-y-4">
                {recentActivity.map((activity, index) => (
                  <div key={index} className="flex items-start gap-3 text-sm group">
                    <div
                      className={cn(
                        "mt-1.5 h-2.5 w-2.5 rounded-full shrink-0 ring-4",
                        activity.type === "success" && "bg-emerald-500 ring-emerald-500/20",
                        activity.type === "info" && "bg-primary ring-primary/20",
                        activity.type === "warning" && "bg-amber-500 ring-amber-500/20",
                        activity.type === "muted" && "bg-muted-foreground/50 ring-muted-foreground/10",
                      )}
                    />
                    <div className="flex-1 min-w-0">
                      <p className="font-medium text-foreground group-hover:text-primary transition-colors">
                        {activity.message}
                      </p>
                      <p className="text-xs text-muted-foreground">{activity.time}</p>
                    </div>
                  </div>
                ))}
              </div>
            </CardContent>
          </Card>
        </section>
      </div>
    </div>
  )
}
