import type { Metadata } from "next"
import { Suspense } from "react"
import { HotelPartnerForm } from "@/components/admin/hotel-partners/hotel-partner-form"
import { Skeleton } from "@/components/ui/skeleton"

export const metadata: Metadata = {
  title: "Add Hotel Partner | Teertham Admin",
  description: "Register a new hotel partner on the Teertham platform with all required documentation and details.",
  openGraph: {
    title: "Add Hotel Partner | Teertham Admin",
    description: "Register a new hotel partner on the Teertham platform.",
    type: "website",
  },
  twitter: {
    card: "summary",
    title: "Add Hotel Partner | Teertham Admin",
    description: "Register a new hotel partner on the Teertham platform.",
  },
  robots: {
    index: false,
    follow: false,
  },
}

function FormSkeleton() {
  return (
    <div className="space-y-6">
      <div className="flex items-center gap-4">
        <Skeleton className="h-10 w-10" />
        <div className="space-y-2">
          <Skeleton className="h-7 w-48" />
          <Skeleton className="h-4 w-72" />
        </div>
      </div>
      {Array.from({ length: 4 }).map((_, i) => (
        <Skeleton key={i} className="h-48 w-full" />
      ))}
    </div>
  )
}

interface PageProps {
  searchParams: Promise<{ id?: string }>
}

export default async function AddHotelPartnerPage({ searchParams }: PageProps) {
  const params = await searchParams
  const partnerId = params.id

  return (
    <main>
      <Suspense fallback={<FormSkeleton />}>
        <HotelPartnerForm partnerId={partnerId} />
      </Suspense>

      {/* JSON-LD Schema */}
      <script
        type="application/ld+json"
        dangerouslySetInnerHTML={{
          __html: JSON.stringify({
            "@context": "https://schema.org",
            "@type": "WebPage",
            name: partnerId ? "Edit Hotel Partner" : "Add Hotel Partner",
            description: "Admin page for managing hotel partner registration on Teertham platform",
            isPartOf: {
              "@type": "WebApplication",
              name: "Teertham Admin",
            },
          }),
        }}
      />
    </main>
  )
}
