import { generatePageMetadata } from "@/lib/seo/metadata"
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card"
import { Bell, CheckCircle2, AlertCircle, Info, Clock } from "lucide-react"
import type { Metadata } from "next"
import { cn } from "@/lib/utils"

export const metadata: Metadata = generatePageMetadata({
  title: "Notifications",
  description: "View and manage your admin notifications",
  path: "/admin/notification",
  noIndex: true,
})

const notifications = [
  {
    id: 1,
    type: "success",
    title: "New user registered",
    message: "A new customer has successfully registered on the platform",
    time: "5 minutes ago",
    read: false,
  },
  {
    id: 2,
    type: "info",
    title: "Agent application submitted",
    message: "New agent application from John Doe is pending review",
    time: "15 minutes ago",
    read: false,
  },
  {
    id: 3,
    type: "success",
    title: "Content updated",
    message: "Hotel listing 'Grand Paradise Resort' has been updated",
    time: "1 hour ago",
    read: false,
  },
  {
    id: 4,
    type: "warning",
    title: "Customer blocked",
    message: "Customer account has been blocked due to policy violation",
    time: "2 hours ago",
    read: true,
  },
  {
    id: 5,
    type: "info",
    title: "System backup completed",
    message: "Automated system backup has been completed successfully",
    time: "3 hours ago",
    read: true,
  },
  {
    id: 6,
    type: "success",
    title: "Payment received",
    message: "Payment of ₹25,000 received from agent commission",
    time: "5 hours ago",
    read: true,
  },
]

export default function NotificationPage() {
  return (
    <div className="admin-section">
      <header className="admin-page-header">
        <div className="flex items-center gap-3">
          <div className="p-2.5 rounded-xl bg-gradient-to-br from-primary/20 to-primary/5">
            <Bell className="h-5 w-5 text-primary" />
          </div>
          <div>
            <h1 className="admin-page-title">Notifications</h1>
            <p className="admin-page-description">Stay updated with your platform activities</p>
          </div>
        </div>
      </header>

      <Card className="admin-card rounded-xl border-0">
        <CardHeader>
          <CardTitle>All Notifications</CardTitle>
          <CardDescription>Recent alerts and updates from your admin panel</CardDescription>
        </CardHeader>
        <CardContent>
          <div className="space-y-3">
            {notifications.map((notification) => {
              const icons = {
                success: CheckCircle2,
                warning: AlertCircle,
                info: Info,
              }
              const Icon = icons[notification.type as keyof typeof icons]

              return (
                <div
                  key={notification.id}
                  className={cn(
                    "flex gap-4 rounded-lg border p-4 transition-all hover:shadow-sm",
                    notification.read ? "bg-muted/30 border-border/50" : "bg-card border-border shadow-sm",
                  )}
                >
                  <div
                    className={cn(
                      "mt-1 shrink-0 rounded-lg p-2",
                      notification.type === "success" && "bg-emerald-500/10",
                      notification.type === "warning" && "bg-amber-500/10",
                      notification.type === "info" && "bg-primary/10",
                    )}
                  >
                    <Icon
                      className={cn(
                        "h-4 w-4",
                        notification.type === "success" && "text-emerald-600",
                        notification.type === "warning" && "text-amber-600",
                        notification.type === "info" && "text-primary",
                      )}
                    />
                  </div>
                  <div className="flex-1 min-w-0">
                    <div className="flex items-start justify-between gap-2 mb-1">
                      <h3
                        className={cn(
                          "font-semibold text-sm",
                          notification.read ? "text-muted-foreground" : "text-foreground",
                        )}
                      >
                        {notification.title}
                      </h3>
                      {!notification.read && (
                        <span className="h-2 w-2 rounded-full bg-primary shrink-0 mt-1.5" aria-label="Unread" />
                      )}
                    </div>
                    <p className="text-sm text-muted-foreground mb-2">{notification.message}</p>
                    <div className="flex items-center gap-1.5 text-xs text-muted-foreground">
                      <Clock className="h-3 w-3" />
                      {notification.time}
                    </div>
                  </div>
                </div>
              )
            })}
          </div>
        </CardContent>
      </Card>
    </div>
  )
}
