"use client"

import { useEffect, useState } from "react"
import { useRouter } from "next/navigation"
import { AgentHeader } from "@/components/agent/agent-header"
import { getClientSession, validateSessionData } from "@/lib/auth/client-session"
import { Card, CardHeader, CardTitle, CardDescription } from "@/components/ui/card"
import { Car, Hotel, Compass, Package, ChevronRight } from "lucide-react"
import Link from "next/link"

export default function BookNewServicePage() {
  const router = useRouter()
  const [agentEmail, setAgentEmail] = useState<string>("agent@teertham.com")
  const [agentName, setAgentName] = useState<string>("Agent")
  const [isLoading, setIsLoading] = useState(true)

  useEffect(() => {
    const session = getClientSession("agent_session")
    if (!session || !validateSessionData(session) || session.role !== "agent") {
      router.push("/login")
      return
    }
    setAgentEmail(session.email || "agent@teertham.com")
    setAgentName(session.name || "Agent")
    setIsLoading(false)
  }, [router])

  if (isLoading) {
    return (
      <div className="flex h-screen items-center justify-center">
        <div className="animate-spin rounded-full h-8 w-8 border-b-2 border-teal-500" />
      </div>
    )
  }

  const bookingServices = [
    {
      title: "Vehicle Booking",
      description: "Book vehicles for your clients' transportation needs",
      href: "/agent/vehicle-booking",
      icon: Car,
      color: "from-teal-500 to-teal-600",
    },
    {
      title: "Hotel Booking",
      description: "Reserve hotels and accommodations for travelers",
      href: "/agent/hotel-booking",
      icon: Hotel,
      color: "from-orange-500 to-orange-600",
    },
    {
      title: "Spiritual Packages",
      description: "Book spiritual journey packages for pilgrims",
      href: "/agent/spiritual-booking",
      icon: Compass,
      color: "from-cyan-500 to-cyan-600",
    },
    {
      title: "Custom Packages",
      description: "Create and book customized travel packages",
      href: "/agent/custom-booking",
      icon: Package,
      color: "from-emerald-500 to-emerald-600",
    },
  ]

  return (
    <div className="min-h-screen bg-background">
      <AgentHeader agentEmail={agentEmail} agentName={agentName} />
      <main className="container mx-auto p-4 md:p-6 lg:p-8">
        <div className="mb-8">
          <h1 className="text-2xl font-bold">Book New Service</h1>
          <p className="text-muted-foreground">Select a service type to create a new booking</p>
        </div>

        <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
          {bookingServices.map((service) => (
            <Link key={service.href} href={service.href}>
              <Card className="border-border/50 hover:shadow-lg transition-all cursor-pointer group h-full">
                <CardHeader>
                  <div className="flex items-center justify-between">
                    <div className={`p-3 rounded-xl bg-gradient-to-br ${service.color} text-white`}>
                      <service.icon className="h-6 w-6" />
                    </div>
                    <ChevronRight className="h-5 w-5 text-muted-foreground group-hover:text-foreground group-hover:translate-x-1 transition-all" />
                  </div>
                  <CardTitle className="mt-4">{service.title}</CardTitle>
                  <CardDescription>{service.description}</CardDescription>
                </CardHeader>
              </Card>
            </Link>
          ))}
        </div>
      </main>
    </div>
  )
}
