"use client"

import { useEffect, useState } from "react"
import { useRouter } from "next/navigation"
import { AgentHeader } from "@/components/agent/agent-header"
import { getClientSession, validateSessionData } from "@/lib/auth/client-session"
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card"
import { Button } from "@/components/ui/button"
import { Badge } from "@/components/ui/badge"
import Link from "next/link"
import {
  Car,
  Hotel,
  Compass,
  Package,
  TrendingUp,
  Calendar,
  DollarSign,
  Users,
  ArrowUpRight,
  Clock,
  CheckCircle2,
  Wallet,
  ArrowRight,
  Activity,
  CreditCard,
  BarChart3,
  Sparkles,
  Target,
  Award,
} from "lucide-react"

export default function AgentDashboardPage() {
  const router = useRouter()
  const [agentEmail, setAgentEmail] = useState<string>("agent@teertham.com")
  const [agentName, setAgentName] = useState<string>("Agent")
  const [isLoading, setIsLoading] = useState(true)

  useEffect(() => {
    const session = getClientSession("agent_session")
    if (!session || !validateSessionData(session) || session.role !== "agent") {
      router.push("/login")
      return
    }
    setAgentEmail(session.email || "agent@teertham.com")
    setAgentName(session.name || "Agent")
    setIsLoading(false)
  }, [router])

  if (isLoading) {
    return (
      <div className="flex h-screen items-center justify-center">
        <div className="animate-spin rounded-full h-8 w-8 border-b-2 border-gray-800" />
      </div>
    )
  }

  const stats = [
    {
      title: "Total Bookings",
      value: "156",
      icon: Calendar,
      change: "+12%",
      gradient: "from-violet-500 to-purple-600",
    },
    { title: "This Month", value: "24", icon: TrendingUp, change: "+8%", gradient: "from-blue-500 to-cyan-500" },
    {
      title: "Total Earnings",
      value: "₹45,200",
      icon: DollarSign,
      change: "+15%",
      gradient: "from-emerald-500 to-teal-500",
    },
    { title: "Active Clients", value: "18", icon: Users, change: "+3", gradient: "from-orange-500 to-amber-500" },
  ]

  const services = [
    {
      title: "Vehicle Bookings",
      count: 45,
      pending: 5,
      icon: Car,
      href: "/agent/all-bookings?tab=vehicle",
      color: "bg-blue-500",
    },
    {
      title: "Hotel Bookings",
      count: 38,
      pending: 3,
      icon: Hotel,
      href: "/agent/all-bookings?tab=hotel",
      color: "bg-orange-500",
    },
    {
      title: "Spiritual Packages",
      count: 52,
      pending: 8,
      icon: Compass,
      href: "/agent/all-bookings?tab=spiritual",
      color: "bg-purple-500",
    },
    {
      title: "Custom Packages",
      count: 21,
      pending: 2,
      icon: Package,
      href: "/agent/all-bookings?tab=custom",
      color: "bg-teal-500",
    },
  ]

  const recentBookings = [
    { id: "BK001", customer: "Rahul Sharma", service: "Vehicle", amount: 4500, status: "upcoming", date: "Dec 15" },
    { id: "BK002", customer: "Ananya Gupta", service: "Hotel", amount: 12500, status: "ongoing", date: "Dec 09" },
    { id: "BK003", customer: "Priya Patel", service: "Spiritual", amount: 8900, status: "completed", date: "Dec 01" },
    { id: "BK004", customer: "Amit Kumar", service: "Custom", amount: 15000, status: "upcoming", date: "Dec 18" },
  ]

  const quickActions = [
    { label: "New Vehicle Booking", href: "/agent/vehicle-booking", icon: Car, gradient: "from-blue-500 to-blue-600" },
    {
      label: "New Hotel Booking",
      href: "/agent/hotel-booking",
      icon: Hotel,
      gradient: "from-orange-500 to-orange-600",
    },
    {
      label: "View All Bookings",
      href: "/agent/all-bookings",
      icon: Calendar,
      gradient: "from-purple-500 to-purple-600",
    },
    {
      label: "Withdraw Earnings",
      href: "/agent/earning-withdraw",
      icon: Wallet,
      gradient: "from-emerald-500 to-emerald-600",
    },
  ]

  const getStatusBadge = (status: string) => {
    switch (status) {
      case "upcoming":
        return <Badge className="bg-blue-100 text-blue-700 border-blue-200">Upcoming</Badge>
      case "ongoing":
        return <Badge className="bg-amber-100 text-amber-700 border-amber-200">Ongoing</Badge>
      case "completed":
        return <Badge className="bg-emerald-100 text-emerald-700 border-emerald-200">Completed</Badge>
      case "cancelled":
        return <Badge className="bg-red-100 text-red-700 border-red-200">Cancelled</Badge>
      default:
        return null
    }
  }

  return (
    <div className="min-h-screen bg-gradient-to-br from-slate-50 via-gray-50 to-zinc-100 dark:from-slate-950 dark:via-gray-950 dark:to-zinc-950">
      <AgentHeader agentEmail={agentEmail} agentName={agentName} />
      <main className="container mx-auto p-4 md:p-6 lg:p-8">
        <div className="relative mb-8 p-6 rounded-2xl bg-gradient-to-r from-slate-900 via-gray-900 to-zinc-800 overflow-hidden">
          <div className="absolute inset-0 bg-[url('data:image/svg+xml;base64,PHN2ZyB3aWR0aD0iNjAiIGhlaWdodD0iNjAiIHZpZXdCb3g9IjAgMCA2MCA2MCIgeG1sbnM9Imh0dHA6Ly93d3cudzMub3JnLzIwMDAvc3ZnIj48ZyBmaWxsPSJub25lIiBmaWxsLXJ1bGU9ImV2ZW5vZGQiPjxwYXRoIGQ9Ik0zNiAxOGMzLjMxNCAwIDYgMi42ODYgNiA2cy0yLjY4NiA2LTYgNi02LTIuNjg2LTYtNiAyLjY4Ni02IDYtNiIgc3Ryb2tlPSJyZ2JhKDI1NSwyNTUsMjU1LDAuMSkiIHN0cm9rZS13aWR0aD0iMiIvPjwvZz48L3N2Zz4=')] opacity-20" />
          <div className="absolute top-0 right-0 w-64 h-64 bg-gradient-to-br from-violet-500/20 to-transparent rounded-full blur-3xl" />
          <div className="absolute bottom-0 left-0 w-48 h-48 bg-gradient-to-tr from-blue-500/20 to-transparent rounded-full blur-3xl" />
          <div className="relative z-10">
            <div className="flex items-center gap-2 mb-2">
              <Sparkles className="h-5 w-5 text-amber-400" />
              <span className="text-amber-400 text-sm font-medium">Agent Dashboard</span>
            </div>
            <h1 className="text-3xl md:text-4xl font-bold text-white mb-2">Welcome back, {agentName}!</h1>
            <p className="text-gray-300 max-w-xl">
              Track your bookings, manage clients, and grow your business with our comprehensive dashboard.
            </p>
          </div>
        </div>

        <div className="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-4 gap-4 mb-8">
          {stats.map((stat) => (
            <Card
              key={stat.title}
              className="group relative overflow-hidden border-0 shadow-lg hover:shadow-xl transition-all duration-300 hover:-translate-y-1"
            >
              <div
                className={`absolute inset-0 bg-gradient-to-br ${stat.gradient} opacity-[0.03] group-hover:opacity-[0.06] transition-opacity`}
              />
              <CardContent className="p-5 relative">
                <div className="flex items-start justify-between">
                  <div>
                    <p className="text-sm text-muted-foreground font-medium">{stat.title}</p>
                    <p className="text-3xl font-bold mt-1">{stat.value}</p>
                    <div className="flex items-center gap-1 mt-2">
                      <div className={`p-0.5 rounded bg-gradient-to-r ${stat.gradient}`}>
                        <ArrowUpRight className="h-3 w-3 text-white" />
                      </div>
                      <span className="text-xs text-emerald-600 font-medium">{stat.change}</span>
                      <span className="text-xs text-muted-foreground">vs last month</span>
                    </div>
                  </div>
                  <div className={`p-3 rounded-xl bg-gradient-to-br ${stat.gradient} shadow-lg`}>
                    <stat.icon className="h-6 w-6 text-white" />
                  </div>
                </div>
              </CardContent>
            </Card>
          ))}
        </div>

        <div className="mb-8">
          <h2 className="text-lg font-semibold mb-4 flex items-center gap-2">
            <Target className="h-5 w-5 text-gray-600" />
            Quick Actions
          </h2>
          <div className="grid grid-cols-2 md:grid-cols-4 gap-3">
            {quickActions.map((action) => (
              <Link key={action.href} href={action.href}>
                <div className="group p-4 rounded-xl border border-border/50 bg-white dark:bg-gray-900 hover:shadow-lg transition-all duration-300 hover:-translate-y-1 cursor-pointer">
                  <div
                    className={`w-12 h-12 rounded-xl bg-gradient-to-br ${action.gradient} flex items-center justify-center mb-3 group-hover:scale-110 transition-transform shadow-lg`}
                  >
                    <action.icon className="h-6 w-6 text-white" />
                  </div>
                  <span className="text-sm font-medium">{action.label}</span>
                </div>
              </Link>
            ))}
          </div>
        </div>

        <div className="grid grid-cols-1 lg:grid-cols-3 gap-6">
          {/* Services Overview */}
          <div className="lg:col-span-2">
            <Card className="border-0 shadow-lg">
              <CardHeader className="pb-2">
                <div className="flex items-center justify-between">
                  <CardTitle className="text-lg font-semibold flex items-center gap-2">
                    <BarChart3 className="h-5 w-5 text-gray-600" />
                    Service Overview
                  </CardTitle>
                  <Link href="/agent/all-bookings">
                    <Button variant="ghost" size="sm" className="text-sm">
                      View All <ArrowRight className="h-4 w-4 ml-1" />
                    </Button>
                  </Link>
                </div>
              </CardHeader>
              <CardContent>
                <div className="grid grid-cols-1 sm:grid-cols-2 gap-4">
                  {services.map((service) => (
                    <Link key={service.title} href={service.href}>
                      <div className="group p-4 rounded-xl border border-border/50 bg-gradient-to-br from-white to-gray-50/50 dark:from-gray-900 dark:to-gray-800/50 hover:shadow-lg transition-all duration-300 hover:-translate-y-1">
                        <div className="flex items-center gap-3">
                          <div
                            className={`p-2.5 rounded-xl ${service.color} shadow-lg group-hover:scale-110 transition-transform`}
                          >
                            <service.icon className="h-5 w-5 text-white" />
                          </div>
                          <div className="flex-1">
                            <p className="font-medium text-sm">{service.title}</p>
                            <div className="flex items-center gap-3 mt-1">
                              <span className="text-2xl font-bold">{service.count}</span>
                              {service.pending > 0 && (
                                <Badge variant="secondary" className="text-xs bg-amber-100 text-amber-700">
                                  {service.pending} pending
                                </Badge>
                              )}
                            </div>
                          </div>
                        </div>
                      </div>
                    </Link>
                  ))}
                </div>
              </CardContent>
            </Card>
          </div>

          <div>
            <Card className="border-0 shadow-lg overflow-hidden">
              <div className="h-32 bg-gradient-to-br from-slate-800 via-gray-800 to-zinc-900 flex items-center justify-center relative overflow-hidden">
                <div className="absolute inset-0 bg-[url('data:image/svg+xml;base64,PHN2ZyB3aWR0aD0iNDAiIGhlaWdodD0iNDAiIHZpZXdCb3g9IjAgMCA0MCA0MCIgeG1sbnM9Imh0dHA6Ly93d3cudzMub3JnLzIwMDAvc3ZnIj48ZyBmaWxsPSJub25lIiBmaWxsLXJ1bGU9ImV2ZW5vZGQiPjxjaXJjbGUgc3Ryb2tlPSJyZ2JhKDI1NSwyNTUsMjU1LDAuMSkiIGN4PSIyMCIgY3k9IjIwIiByPSI1Ii8+PC9nPjwvc3ZnPg==')] opacity-30" />
                <div className="absolute top-0 right-0 w-32 h-32 bg-gradient-to-bl from-emerald-500/20 to-transparent rounded-full blur-2xl" />
                <div className="text-center relative z-10">
                  <div className="flex items-center justify-center gap-2 mb-1">
                    <Wallet className="h-4 w-4 text-emerald-400" />
                    <p className="text-white/70 text-sm">Available Balance</p>
                  </div>
                  <p className="text-4xl font-bold text-white">₹12,450</p>
                </div>
              </div>
              <CardContent className="p-4">
                <div className="space-y-3">
                  <div className="flex items-center justify-between py-2 border-b border-border/50">
                    <div className="flex items-center gap-2">
                      <Clock className="h-4 w-4 text-amber-500" />
                      <span className="text-sm text-muted-foreground">Pending</span>
                    </div>
                    <span className="font-semibold text-amber-600">₹5,000</span>
                  </div>
                  <div className="flex items-center justify-between py-2 border-b border-border/50">
                    <div className="flex items-center gap-2">
                      <TrendingUp className="h-4 w-4 text-emerald-500" />
                      <span className="text-sm text-muted-foreground">This Month</span>
                    </div>
                    <span className="font-semibold text-emerald-600">+₹8,500</span>
                  </div>
                  <Link href="/agent/earning-withdraw">
                    <Button className="w-full bg-gradient-to-r from-slate-800 to-gray-900 hover:from-slate-700 hover:to-gray-800 mt-2 shadow-lg">
                      <CreditCard className="h-4 w-4 mr-2" />
                      Withdraw Funds
                    </Button>
                  </Link>
                </div>
              </CardContent>
            </Card>
          </div>
        </div>

        {/* Recent Bookings */}
        <Card className="mt-6 border-0 shadow-lg">
          <CardHeader className="pb-2">
            <div className="flex items-center justify-between">
              <CardTitle className="text-lg font-semibold flex items-center gap-2">
                <Activity className="h-5 w-5 text-gray-600" />
                Recent Bookings
              </CardTitle>
              <Link href="/agent/all-bookings">
                <Button variant="ghost" size="sm" className="text-sm">
                  View All <ArrowRight className="h-4 w-4 ml-1" />
                </Button>
              </Link>
            </div>
          </CardHeader>
          <CardContent>
            <div className="space-y-3">
              {recentBookings.map((booking) => (
                <div
                  key={booking.id}
                  className="flex items-center justify-between p-4 rounded-xl bg-gradient-to-r from-gray-50 to-white dark:from-gray-900 dark:to-gray-800 border border-border/30 hover:shadow-md transition-all"
                >
                  <div className="flex items-center gap-3">
                    <div className="w-12 h-12 rounded-full bg-gradient-to-br from-violet-500 to-purple-600 flex items-center justify-center shadow-lg">
                      <span className="text-sm font-bold text-white">{booking.customer.charAt(0)}</span>
                    </div>
                    <div>
                      <p className="font-medium">{booking.customer}</p>
                      <p className="text-sm text-muted-foreground">
                        {booking.service} • {booking.date}
                      </p>
                    </div>
                  </div>
                  <div className="flex items-center gap-4">
                    {getStatusBadge(booking.status)}
                    <span className="font-bold text-lg">₹{booking.amount.toLocaleString("en-IN")}</span>
                  </div>
                </div>
              ))}
            </div>
          </CardContent>
        </Card>

        <div className="grid grid-cols-1 md:grid-cols-3 gap-4 mt-6">
          <Card className="border-0 shadow-lg overflow-hidden">
            <div className="h-1 bg-gradient-to-r from-emerald-500 to-teal-500" />
            <CardContent className="p-4 flex items-center gap-4">
              <div className="p-3 rounded-xl bg-gradient-to-br from-emerald-500 to-teal-500 shadow-lg">
                <CheckCircle2 className="h-6 w-6 text-white" />
              </div>
              <div>
                <p className="text-2xl font-bold">142</p>
                <p className="text-sm text-muted-foreground">Completed Bookings</p>
              </div>
            </CardContent>
          </Card>
          <Card className="border-0 shadow-lg overflow-hidden">
            <div className="h-1 bg-gradient-to-r from-amber-500 to-orange-500" />
            <CardContent className="p-4 flex items-center gap-4">
              <div className="p-3 rounded-xl bg-gradient-to-br from-amber-500 to-orange-500 shadow-lg">
                <Clock className="h-6 w-6 text-white" />
              </div>
              <div>
                <p className="text-2xl font-bold">8</p>
                <p className="text-sm text-muted-foreground">Pending Bookings</p>
              </div>
            </CardContent>
          </Card>
          <Card className="border-0 shadow-lg overflow-hidden">
            <div className="h-1 bg-gradient-to-r from-yellow-500 to-amber-500" />
            <CardContent className="p-4 flex items-center gap-4">
              <div className="p-3 rounded-xl bg-gradient-to-br from-yellow-500 to-amber-500 shadow-lg">
                <Award className="h-6 w-6 text-white" />
              </div>
              <div>
                <p className="text-2xl font-bold">4.8</p>
                <p className="text-sm text-muted-foreground">Average Rating</p>
              </div>
            </CardContent>
          </Card>
        </div>
      </main>
    </div>
  )
}
