"use client"

import { useEffect, useState, useMemo } from "react"
import { useRouter } from "next/navigation"
import { AgentHeader } from "@/components/agent/agent-header"
import { getClientSession, validateSessionData } from "@/lib/auth/client-session"
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card"
import { Button } from "@/components/ui/button"
import { Label } from "@/components/ui/label"
import { Input } from "@/components/ui/input"
import { Badge } from "@/components/ui/badge"
import { Calendar } from "@/components/ui/calendar"
import { Popover, PopoverContent, PopoverTrigger } from "@/components/ui/popover"
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select"
import { toast } from "sonner"
import { format } from "date-fns"
import { Car, CalendarIcon, MapPin, Users, CheckCircle2, Fuel, Sparkles, UserPlus } from "lucide-react"
import { cn } from "@/lib/utils"

// Mock data for cities
const cities = [
  { id: "delhi", name: "Delhi" },
  { id: "jaipur", name: "Jaipur" },
  { id: "agra", name: "Agra" },
  { id: "varanasi", name: "Varanasi" },
  { id: "haridwar", name: "Haridwar" },
  { id: "rishikesh", name: "Rishikesh" },
  { id: "mathura", name: "Mathura" },
  { id: "vrindavan", name: "Vrindavan" },
  { id: "ayodhya", name: "Ayodhya" },
  { id: "lucknow", name: "Lucknow" },
]

// Mock data for vehicles with routes
const allVehicles = [
  {
    id: "v1",
    name: "Swift Dzire",
    type: "Sedan",
    image: "/placeholder.svg?height=120&width=200",
    perDayRate: 2500,
    maxPassengers: 4,
    features: ["AC", "Music System", "GPS"],
    fuelType: "Petrol",
    routes: ["delhi-jaipur", "delhi-agra", "jaipur-agra", "delhi-haridwar", "delhi-rishikesh"],
  },
  {
    id: "v2",
    name: "Toyota Innova",
    type: "SUV",
    image: "/placeholder.svg?height=120&width=200",
    perDayRate: 4500,
    maxPassengers: 7,
    features: ["AC", "Music System", "GPS", "Extra Luggage Space"],
    fuelType: "Diesel",
    routes: ["delhi-jaipur", "delhi-agra", "delhi-varanasi", "delhi-haridwar", "delhi-rishikesh", "jaipur-agra"],
  },
  {
    id: "v3",
    name: "Tempo Traveller",
    type: "Traveller",
    image: "/placeholder.svg?height=120&width=200",
    perDayRate: 6500,
    maxPassengers: 12,
    features: ["AC", "Music System", "GPS", "Pushback Seats"],
    fuelType: "Diesel",
    routes: ["delhi-jaipur", "delhi-agra", "delhi-varanasi", "delhi-haridwar", "delhi-ayodhya", "delhi-lucknow"],
  },
  {
    id: "v4",
    name: "Luxury Tempo Traveller",
    type: "Luxury Traveller",
    image: "/placeholder.svg?height=120&width=200",
    perDayRate: 9000,
    maxPassengers: 17,
    features: ["AC", "Music System", "GPS", "Pushback Seats", "LCD TV", "Ice Box"],
    fuelType: "Diesel",
    routes: ["delhi-jaipur", "delhi-agra", "delhi-varanasi", "delhi-haridwar", "delhi-ayodhya", "delhi-lucknow"],
  },
  {
    id: "v5",
    name: "Maruti Ertiga",
    type: "MPV",
    image: "/placeholder.svg?height=120&width=200",
    perDayRate: 3200,
    maxPassengers: 6,
    features: ["AC", "Music System", "GPS"],
    fuelType: "Petrol/CNG",
    routes: ["delhi-agra", "delhi-mathura", "delhi-vrindavan", "agra-mathura", "mathura-vrindavan"],
  },
  {
    id: "v6",
    name: "Toyota Fortuner",
    type: "Premium SUV",
    image: "/placeholder.svg?height=120&width=200",
    perDayRate: 7500,
    maxPassengers: 6,
    features: ["AC", "Premium Music System", "GPS", "Leather Seats", "Sunroof"],
    fuelType: "Diesel",
    routes: ["delhi-jaipur", "delhi-agra", "delhi-rishikesh", "delhi-haridwar"],
  },
]

interface Passenger {
  id: string
  name: string
  age: string
  gender: string
}

export default function VehicleBookingPage() {
  const router = useRouter()
  const [agentEmail, setAgentEmail] = useState<string>("agent@teertham.com")
  const [agentName, setAgentName] = useState<string>("Agent")
  const [isLoading, setIsLoading] = useState(true)

  // Booking form state
  const [fromCity, setFromCity] = useState<string>("")
  const [toCity, setToCity] = useState<string>("")
  const [bookingDate, setBookingDate] = useState<Date>()
  const [passengers, setPassengers] = useState<string>("")
  const [selectedVehicle, setSelectedVehicle] = useState<string | null>(null)
  const [pickupTime, setPickupTime] = useState<string>("")

  const [passengerDetails, setPassengerDetails] = useState<Passenger[]>([])

  useEffect(() => {
    const session = getClientSession("agent_session")
    if (!session || !validateSessionData(session) || session.role !== "agent") {
      router.push("/login")
      return
    }
    setAgentEmail(session.email || "agent@teertham.com")
    setAgentName(session.name || "Agent")
    setIsLoading(false)
  }, [router])

  useEffect(() => {
    if (passengers) {
      const count = Number.parseInt(passengers)
      const currentCount = passengerDetails.length
      if (count > currentCount) {
        const newPassengers = Array.from({ length: count - currentCount }, (_, i) => ({
          id: `p-${Date.now()}-${i}`,
          name: "",
          age: "",
          gender: "",
        }))
        setPassengerDetails([...passengerDetails, ...newPassengers])
      } else if (count < currentCount) {
        setPassengerDetails(passengerDetails.slice(0, count))
      }
    }
  }, [passengers])

  // Filter vehicles based on route and passenger count
  const availableVehicles = useMemo(() => {
    if (!fromCity || !toCity) return []

    const route = `${fromCity}-${toCity}`
    const reverseRoute = `${toCity}-${fromCity}`
    const passengerCount = passengers ? Number.parseInt(passengers) : 0

    return allVehicles.filter((vehicle) => {
      const routeMatch = vehicle.routes.includes(route) || vehicle.routes.includes(reverseRoute)
      const passengerMatch = passengerCount ? vehicle.maxPassengers >= passengerCount : true
      return routeMatch && passengerMatch
    })
  }, [fromCity, toCity, passengers])

  const selectedVehicleData = useMemo(() => {
    return allVehicles.find((v) => v.id === selectedVehicle)
  }, [selectedVehicle])

  const updatePassengerDetail = (id: string, field: keyof Passenger, value: string) => {
    setPassengerDetails(passengerDetails.map((p) => (p.id === id ? { ...p, [field]: value } : p)))
  }

  const handleBookNow = () => {
    if (!fromCity || !toCity || !bookingDate || !passengers || !selectedVehicle || !pickupTime) {
      toast.error("Please fill all required fields")
      return
    }

    // Validate passenger details
    const incompletePassenger = passengerDetails.find((p) => !p.name || !p.age || !p.gender)
    if (incompletePassenger) {
      toast.error("Please fill all passenger details")
      return
    }

    toast.success("Vehicle booked successfully!", {
      description: `${selectedVehicleData?.name} booked for ${format(bookingDate, "PPP")}`,
    })
  }

  if (isLoading) {
    return (
      <div className="flex h-screen items-center justify-center">
        <div className="animate-spin rounded-full h-8 w-8 border-b-2 border-gray-800" />
      </div>
    )
  }

  return (
    <div className="min-h-screen bg-gradient-to-br from-slate-50 via-gray-50 to-zinc-100 dark:from-slate-950 dark:via-gray-950 dark:to-zinc-950">
      <AgentHeader agentEmail={agentEmail} agentName={agentName} />
      <main className="container mx-auto p-4 md:p-6 lg:p-8">
        <div className="relative mb-8 p-6 rounded-2xl bg-gradient-to-r from-gray-900 via-gray-800 to-zinc-900 overflow-hidden shadow-xl">
          <div className="absolute inset-0 bg-[url('data:image/svg+xml;base64,PHN2ZyB3aWR0aD0iNjAiIGhlaWdodD0iNjAiIHZpZXdCb3g9IjAgMCA2MCA2MCIgeG1sbnM9Imh0dHA6Ly93d3cudzMub3JnLzIwMDAvc3ZnIj48ZyBmaWxsPSJub25lIiBmaWxsLXJ1bGU9ImV2ZW5vZGQiPjxwYXRoIGQ9Ik0zNiAxOGMzLjMxNCAwIDYgMi42ODYgNiA2cy0yLjY4NiA2LTYgNi02LTIuNjg2LTYtNiAyLjY4NiA2IDYtNiIgc3Ryb2tlPSJyZ2JhKDI1NSwyNTUsMjU1LDAuMSkiIHN0cm9rZS13aWR0aD0iMiIvPjwvZz48L3N2Zz4=')] opacity-20" />
          <div className="absolute top-0 right-0 w-64 h-64 bg-white/5 rounded-full blur-3xl -translate-y-1/2 translate-x-1/2" />
          <div className="relative z-10 flex items-center gap-4">
            <div className="p-3 rounded-xl bg-white/10 backdrop-blur-sm border border-white/10">
              <Car className="h-8 w-8 text-white" />
            </div>
            <div>
              <h1 className="text-2xl md:text-3xl font-bold text-white">Vehicle Booking</h1>
              <p className="text-gray-300">Book vehicles for your clients' travel needs</p>
            </div>
          </div>
        </div>

        {/* Booking Form */}
        <Card className="border-0 shadow-lg mb-6">
          <CardHeader>
            <CardTitle className="flex items-center gap-2 text-lg">
              <MapPin className="h-5 w-5 text-gray-700" />
              Select Route & Details
            </CardTitle>
          </CardHeader>
          <CardContent>
            <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-4">
              {/* From City */}
              <div className="space-y-2">
                <Label>From City</Label>
                <Select value={fromCity} onValueChange={setFromCity}>
                  <SelectTrigger>
                    <SelectValue placeholder="Select pickup city" />
                  </SelectTrigger>
                  <SelectContent>
                    {cities.map((city) => (
                      <SelectItem key={city.id} value={city.id} disabled={city.id === toCity}>
                        {city.name}
                      </SelectItem>
                    ))}
                  </SelectContent>
                </Select>
              </div>

              {/* To City */}
              <div className="space-y-2">
                <Label>To City</Label>
                <Select value={toCity} onValueChange={setToCity}>
                  <SelectTrigger>
                    <SelectValue placeholder="Select drop city" />
                  </SelectTrigger>
                  <SelectContent>
                    {cities.map((city) => (
                      <SelectItem key={city.id} value={city.id} disabled={city.id === fromCity}>
                        {city.name}
                      </SelectItem>
                    ))}
                  </SelectContent>
                </Select>
              </div>

              {/* Booking Date */}
              <div className="space-y-2">
                <Label>Booking Date</Label>
                <Popover>
                  <PopoverTrigger asChild>
                    <Button
                      variant="outline"
                      className={cn(
                        "w-full justify-start text-left font-normal",
                        !bookingDate && "text-muted-foreground",
                      )}
                    >
                      <CalendarIcon className="mr-2 h-4 w-4" />
                      {bookingDate ? format(bookingDate, "PPP") : "Select date"}
                    </Button>
                  </PopoverTrigger>
                  <PopoverContent className="w-auto p-0" align="start">
                    <Calendar
                      mode="single"
                      selected={bookingDate}
                      onSelect={setBookingDate}
                      disabled={(date) => date < new Date()}
                      initialFocus
                    />
                  </PopoverContent>
                </Popover>
              </div>

              {/* Total Passengers */}
              <div className="space-y-2">
                <Label>Total Passengers</Label>
                <Select value={passengers} onValueChange={setPassengers}>
                  <SelectTrigger>
                    <SelectValue placeholder="Select passengers" />
                  </SelectTrigger>
                  <SelectContent>
                    {[1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15, 16, 17].map((num) => (
                      <SelectItem key={num} value={num.toString()}>
                        {num} {num === 1 ? "Passenger" : "Passengers"}
                      </SelectItem>
                    ))}
                  </SelectContent>
                </Select>
              </div>
            </div>
          </CardContent>
        </Card>

        {/* Available Vehicles */}
        {fromCity && toCity && (
          <div className="mb-6">
            <h2 className="text-lg font-semibold mb-4 flex items-center gap-2">
              <Car className="h-5 w-5 text-gray-600" />
              Available Vehicles
              {availableVehicles.length > 0 && (
                <Badge variant="secondary" className="ml-2">
                  {availableVehicles.length} found
                </Badge>
              )}
            </h2>

            {availableVehicles.length === 0 ? (
              <Card className="border-0 shadow-lg">
                <CardContent className="py-12 text-center">
                  <Car className="h-12 w-12 text-muted-foreground mx-auto mb-4" />
                  <h3 className="text-lg font-medium mb-2">No vehicles available</h3>
                  <p className="text-muted-foreground">
                    No vehicles found for this route. Try a different route or reduce passengers.
                  </p>
                </CardContent>
              </Card>
            ) : (
              <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-4">
                {availableVehicles.map((vehicle) => (
                  <Card
                    key={vehicle.id}
                    className={cn(
                      "border-2 cursor-pointer transition-all duration-300 hover:shadow-xl hover:-translate-y-1",
                      selectedVehicle === vehicle.id
                        ? "border-gray-800 ring-2 ring-gray-800/20 shadow-lg"
                        : "border-transparent shadow-md",
                    )}
                    onClick={() => setSelectedVehicle(vehicle.id)}
                  >
                    <CardContent className="p-4">
                      <div className="relative">
                        <img
                          src={vehicle.image || "/placeholder.svg"}
                          alt={vehicle.name}
                          className="w-full h-32 object-cover rounded-lg mb-4"
                        />
                        {selectedVehicle === vehicle.id && (
                          <div className="absolute top-2 right-2 p-1.5 bg-gray-900 rounded-full">
                            <CheckCircle2 className="h-4 w-4 text-white" />
                          </div>
                        )}
                        <Badge className="absolute top-2 left-2 bg-black/70">{vehicle.type}</Badge>
                      </div>

                      <h3 className="font-bold text-lg mb-2">{vehicle.name}</h3>

                      <div className="flex items-center gap-4 text-sm text-muted-foreground mb-3">
                        <div className="flex items-center gap-1">
                          <Users className="h-4 w-4" />
                          <span>{vehicle.maxPassengers} seats</span>
                        </div>
                        <div className="flex items-center gap-1">
                          <Fuel className="h-4 w-4" />
                          <span>{vehicle.fuelType}</span>
                        </div>
                      </div>

                      <div className="flex flex-wrap gap-1.5 mb-4">
                        {vehicle.features.slice(0, 3).map((feature) => (
                          <Badge key={feature} variant="secondary" className="text-xs">
                            {feature}
                          </Badge>
                        ))}
                        {vehicle.features.length > 3 && (
                          <Badge variant="secondary" className="text-xs">
                            +{vehicle.features.length - 3}
                          </Badge>
                        )}
                      </div>

                      <div className="flex items-center justify-between pt-3 border-t">
                        <div>
                          <span className="text-2xl font-bold text-gray-800 dark:text-gray-200">
                            ₹{vehicle.perDayRate.toLocaleString("en-IN")}
                          </span>
                          <span className="text-sm text-muted-foreground">/day</span>
                        </div>
                        <Button
                          size="sm"
                          className="bg-gradient-to-r from-gray-900 to-gray-700 hover:from-gray-800 hover:to-gray-600 text-white shadow-md"
                        >
                          {selectedVehicle === vehicle.id ? "Selected" : "Select"}
                        </Button>
                      </div>
                    </CardContent>
                  </Card>
                ))}
              </div>
            )}
          </div>
        )}

        {/* Booking Summary */}
        {selectedVehicle && selectedVehicleData && (
          <Card className="border-0 shadow-lg overflow-hidden">
            <div className="h-1 bg-gradient-to-r from-gray-900 via-gray-700 to-gray-900" />
            <CardHeader>
              <CardTitle className="flex items-center gap-2 text-lg">
                <Sparkles className="h-5 w-5 text-gray-700" />
                Booking Summary
              </CardTitle>
            </CardHeader>
            <CardContent>
              <div className="grid grid-cols-1 lg:grid-cols-2 gap-6">
                <div className="space-y-4">
                  <div className="flex items-center gap-4 p-4 bg-gray-50 dark:bg-gray-800/50 rounded-xl">
                    <img
                      src={selectedVehicleData.image || "/placeholder.svg"}
                      alt={selectedVehicleData.name}
                      className="w-24 h-16 object-cover rounded-lg"
                    />
                    <div>
                      <h4 className="font-bold">{selectedVehicleData.name}</h4>
                      <p className="text-sm text-muted-foreground">{selectedVehicleData.type}</p>
                    </div>
                  </div>

                  <div className="grid grid-cols-2 gap-3">
                    <div className="p-3 bg-gray-50 dark:bg-gray-800/50 rounded-lg">
                      <p className="text-xs text-muted-foreground mb-1">From</p>
                      <p className="font-medium capitalize">{fromCity}</p>
                    </div>
                    <div className="p-3 bg-gray-50 dark:bg-gray-800/50 rounded-lg">
                      <p className="text-xs text-muted-foreground mb-1">To</p>
                      <p className="font-medium capitalize">{toCity}</p>
                    </div>
                    <div className="p-3 bg-gray-50 dark:bg-gray-800/50 rounded-lg">
                      <p className="text-xs text-muted-foreground mb-1">Date</p>
                      <p className="font-medium">{bookingDate ? format(bookingDate, "PP") : "Not selected"}</p>
                    </div>
                    <div className="p-3 bg-gray-50 dark:bg-gray-800/50 rounded-lg">
                      <p className="text-xs text-muted-foreground mb-1">Passengers</p>
                      <p className="font-medium">{passengers || "Not selected"}</p>
                    </div>
                  </div>
                </div>

                <div className="space-y-4">
                  <div className="space-y-2">
                    <Label>Pickup Time</Label>
                    <Select value={pickupTime} onValueChange={setPickupTime}>
                      <SelectTrigger>
                        <SelectValue placeholder="Select pickup time" />
                      </SelectTrigger>
                      <SelectContent>
                        {[
                          "05:00 AM",
                          "06:00 AM",
                          "07:00 AM",
                          "08:00 AM",
                          "09:00 AM",
                          "10:00 AM",
                          "11:00 AM",
                          "12:00 PM",
                          "01:00 PM",
                          "02:00 PM",
                          "03:00 PM",
                          "04:00 PM",
                          "05:00 PM",
                          "06:00 PM",
                        ].map((time) => (
                          <SelectItem key={time} value={time}>
                            {time}
                          </SelectItem>
                        ))}
                      </SelectContent>
                    </Select>
                  </div>

                  <div className="p-4 bg-gradient-to-br from-gray-50 to-zinc-100 dark:from-gray-800/50 dark:to-zinc-800/50 rounded-xl border border-gray-200 dark:border-gray-700">
                    <div className="flex items-center justify-between mb-2">
                      <span className="text-muted-foreground">Base Rate</span>
                      <span>₹{selectedVehicleData.perDayRate.toLocaleString("en-IN")}</span>
                    </div>
                    <div className="flex items-center justify-between mb-2">
                      <span className="text-muted-foreground">GST (5%)</span>
                      <span>₹{Math.round(selectedVehicleData.perDayRate * 0.05).toLocaleString("en-IN")}</span>
                    </div>
                    <div className="flex items-center justify-between mb-2">
                      <span className="text-muted-foreground">Service Fee</span>
                      <span>₹200</span>
                    </div>
                    <div className="h-px bg-gray-200 dark:bg-gray-700 my-3" />
                    <div className="flex items-center justify-between">
                      <span className="font-bold">Total Amount</span>
                      <span className="text-xl font-bold text-gray-800 dark:text-gray-200">
                        ₹
                        {(
                          selectedVehicleData.perDayRate +
                          Math.round(selectedVehicleData.perDayRate * 0.05) +
                          200
                        ).toLocaleString("en-IN")}
                      </span>
                    </div>
                  </div>
                </div>
              </div>

              {passengers && Number.parseInt(passengers) > 0 && (
                <div className="mt-6 pt-6 border-t">
                  <h3 className="text-lg font-semibold mb-4 flex items-center gap-2">
                    <UserPlus className="h-5 w-5 text-gray-700" />
                    Passenger Details
                  </h3>
                  <div className="space-y-4">
                    {passengerDetails.map((passenger, index) => (
                      <div key={passenger.id} className="p-4 bg-gray-50 dark:bg-gray-800/50 rounded-xl">
                        <div className="flex items-center justify-between mb-3">
                          <span className="font-medium">Passenger {index + 1}</span>
                        </div>
                        <div className="grid grid-cols-1 md:grid-cols-3 gap-4">
                          <div className="space-y-2">
                            <Label>Full Name</Label>
                            <Input
                              placeholder="Enter full name"
                              value={passenger.name}
                              onChange={(e) => updatePassengerDetail(passenger.id, "name", e.target.value)}
                            />
                          </div>
                          <div className="space-y-2">
                            <Label>Age</Label>
                            <Input
                              type="number"
                              placeholder="Enter age"
                              value={passenger.age}
                              onChange={(e) => updatePassengerDetail(passenger.id, "age", e.target.value)}
                            />
                          </div>
                          <div className="space-y-2">
                            <Label>Gender</Label>
                            <Select
                              value={passenger.gender}
                              onValueChange={(val) => updatePassengerDetail(passenger.id, "gender", val)}
                            >
                              <SelectTrigger>
                                <SelectValue placeholder="Select gender" />
                              </SelectTrigger>
                              <SelectContent>
                                <SelectItem value="male">Male</SelectItem>
                                <SelectItem value="female">Female</SelectItem>
                                <SelectItem value="other">Other</SelectItem>
                              </SelectContent>
                            </Select>
                          </div>
                        </div>
                      </div>
                    ))}
                  </div>
                </div>
              )}

              <Button
                className="w-full mt-6 h-12 bg-gradient-to-r from-gray-900 to-gray-700 hover:from-gray-800 hover:to-gray-600 text-white shadow-lg text-lg font-semibold"
                onClick={handleBookNow}
              >
                Book Now
              </Button>
            </CardContent>
          </Card>
        )}
      </main>
    </div>
  )
}
