"use client"

import { Dialog, DialogContent, DialogHeader, DialogTitle } from "@/components/ui/dialog"
import { Badge } from "@/components/ui/badge"
import { Separator } from "@/components/ui/separator"
import {
  Calendar,
  Users,
  MapPin,
  IndianRupee,
  User,
  Phone,
  Mail,
  CreditCard,
  Car,
  Building2,
  Package,
} from "lucide-react"
import type { AgentBooking } from "@/lib/data/agent-bookings"
import { customPackageDataStore } from "@/lib/data/custom-packages"

interface ViewAgentBookingDetailsDialogProps {
  booking: AgentBooking
  open: boolean
  onOpenChange: (open: boolean) => void
}

export function ViewAgentBookingDetailsDialog({ booking, open, onOpenChange }: ViewAgentBookingDetailsDialogProps) {
  // Get full package details
  const packageDetails = customPackageDataStore.getPackageById(booking.packageId)

  // Format currency
  const formatCurrency = (amount: number): string => {
    return new Intl.NumberFormat("en-IN", {
      style: "currency",
      currency: "INR",
      maximumFractionDigits: 0,
    }).format(amount)
  }

  // Format date
  const formatDate = (dateString: string): string => {
    return new Date(dateString).toLocaleDateString("en-IN", {
      day: "2-digit",
      month: "short",
      year: "numeric",
    })
  }

  // Get status color
  const getStatusColor = (status: string) => {
    switch (status) {
      case "booked":
        return "bg-blue-500/10 text-blue-600 border-blue-500/20"
      case "ongoing":
        return "bg-amber-500/10 text-amber-600 border-amber-500/20"
      case "completed":
        return "bg-green-500/10 text-green-600 border-green-500/20"
      case "cancelled":
        return "bg-red-500/10 text-red-600 border-red-500/20"
      default:
        return "bg-muted text-muted-foreground"
    }
  }

  return (
    <Dialog open={open} onOpenChange={onOpenChange}>
      <DialogContent className="max-w-4xl max-h-[90vh] overflow-y-auto">
        <DialogHeader>
          <div className="flex items-start justify-between">
            <div>
              <DialogTitle className="text-2xl">Agent Booking Details</DialogTitle>
              <p className="text-sm text-muted-foreground mt-1">Booking ID: {booking.id}</p>
            </div>
            <Badge className={getStatusColor(booking.status)} variant="outline">
              {booking.status.charAt(0).toUpperCase() + booking.status.slice(1)}
            </Badge>
          </div>
        </DialogHeader>

        <div className="space-y-6 mt-4">
          {/* Agent Information */}
          <div className="space-y-3">
            <h3 className="text-lg font-semibold flex items-center gap-2">
              <Building2 className="h-5 w-5 text-primary" />
              Agent Information
            </h3>
            <div className="grid grid-cols-1 md:grid-cols-2 gap-4 bg-muted/50 p-4 rounded-lg">
              <div className="flex items-start gap-3">
                <User className="h-4 w-4 text-muted-foreground mt-1" />
                <div>
                  <p className="text-sm text-muted-foreground">Agent Name</p>
                  <p className="font-medium">{booking.agentName}</p>
                </div>
              </div>
              <div className="flex items-start gap-3">
                <Mail className="h-4 w-4 text-muted-foreground mt-1" />
                <div>
                  <p className="text-sm text-muted-foreground">Email</p>
                  <p className="font-medium">{booking.agentEmail}</p>
                </div>
              </div>
              <div className="flex items-start gap-3">
                <Phone className="h-4 w-4 text-muted-foreground mt-1" />
                <div>
                  <p className="text-sm text-muted-foreground">Phone</p>
                  <p className="font-medium">{booking.agentPhone}</p>
                </div>
              </div>
              <div className="flex items-start gap-3">
                <IndianRupee className="h-4 w-4 text-muted-foreground mt-1" />
                <div>
                  <p className="text-sm text-muted-foreground">Commission</p>
                  <p className="font-medium text-green-600">{formatCurrency(booking.agentCommission)}</p>
                </div>
              </div>
            </div>
          </div>

          <Separator />

          {/* Customer Information */}
          <div className="space-y-3">
            <h3 className="text-lg font-semibold flex items-center gap-2">
              <User className="h-5 w-5 text-primary" />
              Customer Information
            </h3>
            <div className="grid grid-cols-1 md:grid-cols-2 gap-4 bg-muted/50 p-4 rounded-lg">
              <div className="flex items-start gap-3">
                <User className="h-4 w-4 text-muted-foreground mt-1" />
                <div>
                  <p className="text-sm text-muted-foreground">Customer Name</p>
                  <p className="font-medium">{booking.customerName}</p>
                </div>
              </div>
              <div className="flex items-start gap-3">
                <Users className="h-4 w-4 text-muted-foreground mt-1" />
                <div>
                  <p className="text-sm text-muted-foreground">Total Travelers</p>
                  <p className="font-medium">{booking.travelersCount} persons</p>
                </div>
              </div>
            </div>
          </div>

          <Separator />

          {/* Package Information */}
          <div className="space-y-3">
            <h3 className="text-lg font-semibold flex items-center gap-2">
              <Package className="h-5 w-5 text-primary" />
              Package Information
            </h3>
            <div className="grid grid-cols-1 md:grid-cols-2 gap-4 bg-muted/50 p-4 rounded-lg">
              <div className="flex items-start gap-3">
                <MapPin className="h-4 w-4 text-muted-foreground mt-1" />
                <div>
                  <p className="text-sm text-muted-foreground">Package Name</p>
                  <p className="font-medium">{booking.packageCityName}</p>
                </div>
              </div>
              <div className="flex items-start gap-3">
                <Package className="h-4 w-4 text-muted-foreground mt-1" />
                <div>
                  <p className="text-sm text-muted-foreground">Category</p>
                  <Badge variant="secondary">{booking.packageCategory}</Badge>
                </div>
              </div>
              <div className="flex items-start gap-3">
                <Calendar className="h-4 w-4 text-muted-foreground mt-1" />
                <div>
                  <p className="text-sm text-muted-foreground">Total Days</p>
                  <p className="font-medium">{booking.totalDays} days</p>
                </div>
              </div>
              <div className="flex items-start gap-3">
                <Car className="h-4 w-4 text-muted-foreground mt-1" />
                <div>
                  <p className="text-sm text-muted-foreground">Vehicle</p>
                  <p className="font-medium">{booking.selectedVehicle}</p>
                </div>
              </div>
            </div>
          </div>

          <Separator />

          {/* Travel Timeline */}
          <div className="space-y-3">
            <h3 className="text-lg font-semibold flex items-center gap-2">
              <Calendar className="h-5 w-5 text-primary" />
              Travel Timeline
            </h3>
            <div className="grid grid-cols-1 md:grid-cols-3 gap-4 bg-muted/50 p-4 rounded-lg">
              <div className="flex items-start gap-3">
                <Calendar className="h-4 w-4 text-muted-foreground mt-1" />
                <div>
                  <p className="text-sm text-muted-foreground">Booking Date</p>
                  <p className="font-medium">{formatDate(booking.bookingDate)}</p>
                </div>
              </div>
              <div className="flex items-start gap-3">
                <Calendar className="h-4 w-4 text-muted-foreground mt-1" />
                <div>
                  <p className="text-sm text-muted-foreground">Start Date</p>
                  <p className="font-medium">{formatDate(booking.startDate)}</p>
                </div>
              </div>
              <div className="flex items-start gap-3">
                <Calendar className="h-4 w-4 text-muted-foreground mt-1" />
                <div>
                  <p className="text-sm text-muted-foreground">End Date</p>
                  <p className="font-medium">{formatDate(booking.endDate)}</p>
                </div>
              </div>
            </div>
          </div>

          <Separator />

          {/* Payment Details */}
          <div className="space-y-3">
            <h3 className="text-lg font-semibold flex items-center gap-2">
              <CreditCard className="h-5 w-5 text-primary" />
              Payment Details
            </h3>
            <div className="space-y-3 bg-muted/50 p-4 rounded-lg">
              <div className="flex justify-between items-center">
                <span className="text-muted-foreground">Total Booking Amount</span>
                <span className="font-semibold text-lg">{formatCurrency(booking.bookingAmount)}</span>
              </div>
              <Separator />
              <div className="flex justify-between items-center">
                <span className="text-muted-foreground">Advance Paid</span>
                <span className="font-medium text-green-600">{formatCurrency(booking.advanceAmount)}</span>
              </div>
              <div className="flex justify-between items-center">
                <span className="text-muted-foreground">Pending Amount</span>
                <span className="font-medium text-amber-600">{formatCurrency(booking.pendingAmount)}</span>
              </div>
              <Separator />
              <div className="flex justify-between items-center">
                <span className="text-muted-foreground">Agent Commission (10%)</span>
                <span className="font-medium text-primary">{formatCurrency(booking.agentCommission)}</span>
              </div>
            </div>
          </div>

          {/* Package Itinerary */}
          {packageDetails && packageDetails.dayPlans.length > 0 && (
            <>
              <Separator />
              <div className="space-y-3">
                <h3 className="text-lg font-semibold flex items-center gap-2">
                  <MapPin className="h-5 w-5 text-primary" />
                  Package Itinerary
                </h3>
                <div className="space-y-4">
                  {packageDetails.dayPlans.map((day, index) => (
                    <div key={index} className="border rounded-lg p-4 space-y-3">
                      <div className="flex items-center gap-2">
                        <Badge variant="outline">Day {day.day}</Badge>
                      </div>
                      {day.locations.map((location, locIndex) => (
                        <div key={locIndex} className="pl-4 border-l-2 border-muted space-y-2">
                          <p className="font-medium flex items-center gap-2">
                            <MapPin className="h-4 w-4 text-primary" />
                            {location.locationName}
                          </p>
                          {location.itineraryDescription && (
                            <p className="text-sm text-muted-foreground">{location.itineraryDescription}</p>
                          )}
                        </div>
                      ))}
                      {day.vehicles.length > 0 && (
                        <div className="mt-3 pt-3 border-t">
                          <p className="text-sm font-medium mb-2">Available Vehicles:</p>
                          <div className="flex flex-wrap gap-2">
                            {day.vehicles.map((vehicle, vIndex) => (
                              <Badge key={vIndex} variant="secondary" className="text-xs">
                                <Car className="h-3 w-3 mr-1" />
                                {vehicle.vehicleName} - ₹{vehicle.perDayPrice}/day
                              </Badge>
                            ))}
                          </div>
                        </div>
                      )}
                    </div>
                  ))}
                </div>
              </div>
            </>
          )}
        </div>
      </DialogContent>
    </Dialog>
  )
}
