"use client"

import { useState, useMemo } from "react"
import { Tabs, TabsContent, TabsList, TabsTrigger } from "@/components/ui/tabs"
import { CustomerTable } from "./customer-table"
import { CustomerDetailsDialog } from "./customer-details-dialog"
import { BlockCustomerDialog } from "./block-customer-dialog"
import { customerDataStore, type Customer } from "@/lib/data/customers"
import { useToastEnhanced } from "@/hooks/use-toast-enhanced"

export function CustomersClient() {
  const toast = useToastEnhanced()
  const [customers, setCustomers] = useState<Customer[]>(customerDataStore.getAllCustomers())
  const [selectedCustomer, setSelectedCustomer] = useState<Customer | null>(null)
  const [isDetailsOpen, setIsDetailsOpen] = useState(false)
  const [isBlockDialogOpen, setIsBlockDialogOpen] = useState(false)
  const [customerToBlock, setCustomerToBlock] = useState<Customer | null>(null)

  const activeCustomers = useMemo(() => customers.filter((c) => c.status === "active"), [customers])
  const blockedCustomers = useMemo(() => customers.filter((c) => c.status === "blocked"), [customers])

  const handleViewDetails = (customer: Customer) => {
    setSelectedCustomer(customer)
    setIsDetailsOpen(true)
  }

  const handleBlockClick = (customer: Customer) => {
    setCustomerToBlock(customer)
    setIsBlockDialogOpen(true)
  }

  const handleBlockCustomer = (reason: string) => {
    if (!customerToBlock) return

    const updated = customerDataStore.blockCustomer(customerToBlock.id, reason)
    if (updated) {
      setCustomers(customerDataStore.getAllCustomers())
      toast.warning({
        title: "Customer Blocked",
        description: `${customerToBlock.firstName} ${customerToBlock.lastName} has been blocked.`,
      })
    }
    setIsBlockDialogOpen(false)
    setCustomerToBlock(null)
  }

  const handleUnblockCustomer = (customer: Customer) => {
    const updated = customerDataStore.unblockCustomer(customer.id)
    if (updated) {
      setCustomers(customerDataStore.getAllCustomers())
      toast.success({
        title: "Customer Unblocked",
        description: `${customer.firstName} ${customer.lastName} has been restored to active status.`,
      })
    }
  }

  return (
    <section aria-labelledby="customers-heading" className="admin-section">
      <header className="admin-page-header">
        <h1 id="customers-heading" className="admin-page-title">
          Customers
        </h1>
        <p className="admin-page-description">Manage customer accounts and monitor their activity.</p>
      </header>

      <Tabs defaultValue="all" className="w-full">
        <TabsList className="mb-6 h-auto p-1.5 bg-muted/50 rounded-xl border border-border/50">
          <TabsTrigger
            value="all"
            className="rounded-lg data-[state=active]:bg-background data-[state=active]:shadow-sm px-4 py-2"
          >
            All ({customers.length})
          </TabsTrigger>
          <TabsTrigger
            value="active"
            className="rounded-lg data-[state=active]:bg-background data-[state=active]:shadow-sm px-4 py-2"
          >
            Active ({activeCustomers.length})
          </TabsTrigger>
          <TabsTrigger
            value="blocked"
            className="rounded-lg data-[state=active]:bg-background data-[state=active]:shadow-sm px-4 py-2"
          >
            Blocked ({blockedCustomers.length})
          </TabsTrigger>
        </TabsList>

        <TabsContent value="all" className="mt-0">
          <CustomerTable
            customers={customers}
            onViewDetails={handleViewDetails}
            onBlock={handleBlockClick}
            onUnblock={handleUnblockCustomer}
          />
        </TabsContent>

        <TabsContent value="active" className="mt-0">
          <CustomerTable
            customers={activeCustomers}
            onViewDetails={handleViewDetails}
            onBlock={handleBlockClick}
            onUnblock={handleUnblockCustomer}
          />
        </TabsContent>

        <TabsContent value="blocked" className="mt-0">
          <CustomerTable
            customers={blockedCustomers}
            onViewDetails={handleViewDetails}
            onBlock={handleBlockClick}
            onUnblock={handleUnblockCustomer}
          />
        </TabsContent>
      </Tabs>

      <CustomerDetailsDialog customer={selectedCustomer} open={isDetailsOpen} onOpenChange={setIsDetailsOpen} />

      <BlockCustomerDialog
        customer={customerToBlock}
        open={isBlockDialogOpen}
        onOpenChange={setIsBlockDialogOpen}
        onConfirm={handleBlockCustomer}
      />
    </section>
  )
}
