"use client"

import { Dialog, DialogContent, DialogDescription, DialogHeader, DialogTitle } from "@/components/ui/dialog"
import { Badge } from "@/components/ui/badge"
import { Separator } from "@/components/ui/separator"
import { Calendar, User, MapPin, Users, IndianRupee, Car, Clock } from "lucide-react"
import type { FixPackageBooking } from "@/lib/data/fix-packages-bookings"
import { fixPackageDataStore } from "@/lib/data/fix-packages"

interface ViewBookingDetailsDialogProps {
  booking: FixPackageBooking
  open: boolean
  onOpenChange: (open: boolean) => void
}

export function ViewBookingDetailsDialog({ booking, open, onOpenChange }: ViewBookingDetailsDialogProps) {
  // Get full package details
  const packageDetails = fixPackageDataStore.getPackageById(booking.packageId)

  // Format currency
  const formatCurrency = (amount: number): string => {
    return new Intl.NumberFormat("en-IN", {
      style: "currency",
      currency: "INR",
      maximumFractionDigits: 0,
    }).format(amount)
  }

  // Format date
  const formatDate = (dateString: string): string => {
    return new Date(dateString).toLocaleDateString("en-IN", {
      day: "2-digit",
      month: "long",
      year: "numeric",
    })
  }

  // Get status badge variant
  const getStatusVariant = (status: string) => {
    switch (status) {
      case "booked":
        return "default"
      case "ongoing":
        return "secondary"
      case "completed":
        return "outline"
      case "cancelled":
        return "destructive"
      default:
        return "default"
    }
  }

  return (
    <Dialog open={open} onOpenChange={onOpenChange}>
      <DialogContent className="max-w-3xl max-h-[90vh] overflow-y-auto">
        <DialogHeader>
          <DialogTitle className="text-2xl">Booking Details</DialogTitle>
          <DialogDescription>Complete information about this fix package booking</DialogDescription>
        </DialogHeader>

        <div className="space-y-6">
          {/* Status Badge */}
          <div className="flex items-center justify-between">
            <Badge variant={getStatusVariant(booking.status)} className="text-sm px-3 py-1 capitalize">
              {booking.status}
            </Badge>
            <span className="text-sm text-muted-foreground">Booking ID: {booking.id}</span>
          </div>

          <Separator />

          {/* Package Information */}
          <div>
            <h3 className="text-lg font-semibold mb-4 flex items-center gap-2">
              <MapPin className="h-5 w-5 text-primary" />
              Package Information
            </h3>
            <div className="grid gap-3 bg-muted/30 p-4 rounded-lg">
              <div className="grid grid-cols-[140px_1fr] gap-2">
                <span className="text-sm font-medium text-muted-foreground">Package Name:</span>
                <span className="text-sm font-semibold">{booking.packageName}</span>
              </div>
              <div className="grid grid-cols-[140px_1fr] gap-2">
                <span className="text-sm font-medium text-muted-foreground">Location:</span>
                <span className="text-sm">{booking.location}</span>
              </div>
              <div className="grid grid-cols-[140px_1fr] gap-2">
                <span className="text-sm font-medium text-muted-foreground">Duration:</span>
                <span className="text-sm">{booking.totalDays} Days</span>
              </div>
              <div className="grid grid-cols-[140px_1fr] gap-2">
                <span className="text-sm font-medium text-muted-foreground">Vehicle:</span>
                <span className="text-sm">{booking.vehicleSelected}</span>
              </div>
            </div>
          </div>

          {/* Customer Information */}
          <div>
            <h3 className="text-lg font-semibold mb-4 flex items-center gap-2">
              <User className="h-5 w-5 text-primary" />
              Customer Information
            </h3>
            <div className="grid gap-3 bg-muted/30 p-4 rounded-lg">
              <div className="grid grid-cols-[140px_1fr] gap-2">
                <span className="text-sm font-medium text-muted-foreground">Customer Name:</span>
                <span className="text-sm font-semibold">{booking.customerName}</span>
              </div>
              <div className="grid grid-cols-[140px_1fr] gap-2">
                <span className="text-sm font-medium text-muted-foreground">Customer ID:</span>
                <span className="text-sm">{booking.customerId}</span>
              </div>
              <div className="grid grid-cols-[140px_1fr] gap-2">
                <span className="text-sm font-medium text-muted-foreground">Travelers Count:</span>
                <span className="text-sm flex items-center gap-1">
                  <Users className="h-4 w-4" />
                  {booking.travelersCount} person{booking.travelersCount > 1 ? "s" : ""}
                </span>
              </div>
            </div>
          </div>

          {/* Booking Timeline */}
          <div>
            <h3 className="text-lg font-semibold mb-4 flex items-center gap-2">
              <Calendar className="h-5 w-5 text-primary" />
              Booking Timeline
            </h3>
            <div className="grid gap-3 bg-muted/30 p-4 rounded-lg">
              <div className="grid grid-cols-[140px_1fr] gap-2">
                <span className="text-sm font-medium text-muted-foreground">Booking Date:</span>
                <span className="text-sm">{formatDate(booking.bookingDate)}</span>
              </div>
              <div className="grid grid-cols-[140px_1fr] gap-2">
                <span className="text-sm font-medium text-muted-foreground">Travel Start:</span>
                <span className="text-sm font-semibold text-green-600">{formatDate(booking.startDate)}</span>
              </div>
              <div className="grid grid-cols-[140px_1fr] gap-2">
                <span className="text-sm font-medium text-muted-foreground">Travel End:</span>
                <span className="text-sm font-semibold text-red-600">{formatDate(booking.endDate)}</span>
              </div>
            </div>
          </div>

          {/* Payment Information */}
          <div>
            <h3 className="text-lg font-semibold mb-4 flex items-center gap-2">
              <IndianRupee className="h-5 w-5 text-primary" />
              Payment Information
            </h3>
            <div className="grid gap-3 bg-muted/30 p-4 rounded-lg">
              <div className="grid grid-cols-[140px_1fr] gap-2">
                <span className="text-sm font-medium text-muted-foreground">Total Amount:</span>
                <span className="text-sm font-bold text-lg">{formatCurrency(booking.bookingAmount)}</span>
              </div>
              <div className="grid grid-cols-[140px_1fr] gap-2">
                <span className="text-sm font-medium text-muted-foreground">Advance Paid:</span>
                <span className="text-sm text-green-600 font-semibold">{formatCurrency(booking.advanceAmount)}</span>
              </div>
              <div className="grid grid-cols-[140px_1fr] gap-2">
                <span className="text-sm font-medium text-muted-foreground">Pending Amount:</span>
                <span className="text-sm text-amber-600 font-semibold">{formatCurrency(booking.pendingAmount)}</span>
              </div>
            </div>
          </div>

          {/* Package Details */}
          {packageDetails && (
            <div>
              <h3 className="text-lg font-semibold mb-4 flex items-center gap-2">
                <Clock className="h-5 w-5 text-primary" />
                Package Details
              </h3>
              <div className="space-y-4 bg-muted/30 p-4 rounded-lg">
                <div>
                  <span className="text-sm font-medium text-muted-foreground block mb-2">Itinerary:</span>
                  <p className="text-sm leading-relaxed">{packageDetails.itineraryDetails}</p>
                </div>
                {packageDetails.otherDetails && (
                  <div>
                    <span className="text-sm font-medium text-muted-foreground block mb-2">Other Details:</span>
                    <p className="text-sm leading-relaxed">{packageDetails.otherDetails}</p>
                  </div>
                )}
                {packageDetails.vehicles.length > 0 && (
                  <div>
                    <span className="text-sm font-medium text-muted-foreground block mb-2">Available Vehicles:</span>
                    <div className="grid gap-2 mt-2">
                      {packageDetails.vehicles.map((vehicle) => (
                        <div key={vehicle.id} className="flex items-center gap-3 p-3 bg-background rounded-lg border">
                          <Car className="h-4 w-4 text-muted-foreground" />
                          <div className="flex-1">
                            <p className="text-sm font-medium">{vehicle.vehicleName}</p>
                            <p className="text-xs text-muted-foreground">
                              {formatCurrency(vehicle.perDayPrice)}/day • {vehicle.passengerLimit} passengers
                            </p>
                          </div>
                        </div>
                      ))}
                    </div>
                  </div>
                )}
              </div>
            </div>
          )}
        </div>
      </DialogContent>
    </Dialog>
  )
}
