"use client"

import { Dialog, DialogContent, DialogHeader, DialogTitle } from "@/components/ui/dialog"
import { Badge } from "@/components/ui/badge"
import { Separator } from "@/components/ui/separator"
import { Hotel, User, Mail, Phone, MapPin, Calendar, Users, IndianRupee, DoorOpen } from "lucide-react"
import type { HotelBooking } from "@/lib/data/hotel-bookings"

interface ViewBookingDetailsDialogProps {
  booking: HotelBooking
  open: boolean
  onOpenChange: (open: boolean) => void
}

export function ViewBookingDetailsDialog({ booking, open, onOpenChange }: ViewBookingDetailsDialogProps) {
  const formatCurrency = (amount: number): string => {
    return new Intl.NumberFormat("en-IN", {
      style: "currency",
      currency: "INR",
      maximumFractionDigits: 0,
    }).format(amount)
  }

  const formatDate = (dateString: string): string => {
    return new Date(dateString).toLocaleDateString("en-IN", {
      day: "2-digit",
      month: "long",
      year: "numeric",
    })
  }

  const getStatusColor = (status: string) => {
    switch (status) {
      case "booked":
        return "bg-blue-500/10 text-blue-600 border-blue-200"
      case "ongoing":
        return "bg-orange-500/10 text-orange-600 border-orange-200"
      case "completed":
        return "bg-green-500/10 text-green-600 border-green-200"
      case "cancelled":
        return "bg-red-500/10 text-red-600 border-red-200"
      default:
        return "bg-gray-500/10 text-gray-600 border-gray-200"
    }
  }

  return (
    <Dialog open={open} onOpenChange={onOpenChange}>
      <DialogContent className="max-w-3xl max-h-[90vh] overflow-y-auto">
        <DialogHeader>
          <div className="flex items-start justify-between">
            <div>
              <DialogTitle className="text-2xl">Booking Details</DialogTitle>
              <p className="text-sm text-muted-foreground mt-1">Booking ID: {booking.id}</p>
            </div>
            <Badge className={getStatusColor(booking.status)} variant="outline">
              {booking.status.charAt(0).toUpperCase() + booking.status.slice(1)}
            </Badge>
          </div>
        </DialogHeader>

        <div className="space-y-6 mt-4">
          {/* Hotel Information */}
          <div className="space-y-3">
            <h3 className="font-semibold text-lg flex items-center gap-2">
              <Hotel className="h-5 w-5 text-primary" />
              Hotel Information
            </h3>
            <div className="bg-muted/50 rounded-lg p-4 space-y-2">
              <div className="grid grid-cols-1 md:grid-cols-2 gap-3">
                <div>
                  <p className="text-sm text-muted-foreground">Hotel Name</p>
                  <p className="font-medium">{booking.hotelName}</p>
                </div>
                <div>
                  <p className="text-sm text-muted-foreground">Owner Name</p>
                  <p className="font-medium">{booking.hotelOwnerName}</p>
                </div>
                <div>
                  <p className="text-sm text-muted-foreground flex items-center gap-1">
                    <Mail className="h-3 w-3" />
                    Email
                  </p>
                  <p className="font-medium text-sm">{booking.hotelEmail}</p>
                </div>
                <div>
                  <p className="text-sm text-muted-foreground flex items-center gap-1">
                    <Phone className="h-3 w-3" />
                    Phone
                  </p>
                  <p className="font-medium">{booking.hotelPhone}</p>
                </div>
                <div className="md:col-span-2">
                  <p className="text-sm text-muted-foreground flex items-center gap-1">
                    <MapPin className="h-3 w-3" />
                    Location
                  </p>
                  <p className="font-medium">
                    {booking.cityName}, {booking.stateName}
                  </p>
                </div>
              </div>
            </div>
          </div>

          <Separator />

          {/* Customer Information */}
          <div className="space-y-3">
            <h3 className="font-semibold text-lg flex items-center gap-2">
              <User className="h-5 w-5 text-primary" />
              Customer Information
            </h3>
            <div className="bg-muted/50 rounded-lg p-4 space-y-2">
              <div className="grid grid-cols-1 md:grid-cols-2 gap-3">
                <div>
                  <p className="text-sm text-muted-foreground">Customer Name</p>
                  <p className="font-medium">{booking.customerName}</p>
                </div>
                <div>
                  <p className="text-sm text-muted-foreground">Customer ID</p>
                  <p className="font-medium text-sm">{booking.customerId}</p>
                </div>
                <div>
                  <p className="text-sm text-muted-foreground flex items-center gap-1">
                    <Mail className="h-3 w-3" />
                    Email
                  </p>
                  <p className="font-medium text-sm">{booking.customerEmail}</p>
                </div>
                <div>
                  <p className="text-sm text-muted-foreground flex items-center gap-1">
                    <Phone className="h-3 w-3" />
                    Phone
                  </p>
                  <p className="font-medium">{booking.customerPhone}</p>
                </div>
              </div>
            </div>
          </div>

          <Separator />

          {/* Booking Details */}
          <div className="space-y-3">
            <h3 className="font-semibold text-lg flex items-center gap-2">
              <Calendar className="h-5 w-5 text-primary" />
              Booking Details
            </h3>
            <div className="bg-muted/50 rounded-lg p-4 space-y-3">
              <div className="grid grid-cols-1 md:grid-cols-2 gap-3">
                <div>
                  <p className="text-sm text-muted-foreground flex items-center gap-1">
                    <DoorOpen className="h-3 w-3" />
                    Room Type
                  </p>
                  <p className="font-medium">{booking.roomSelected}</p>
                </div>
                <div>
                  <p className="text-sm text-muted-foreground flex items-center gap-1">
                    <Users className="h-3 w-3" />
                    Total Members
                  </p>
                  <p className="font-medium">{booking.totalMembers}</p>
                </div>
                <div>
                  <p className="text-sm text-muted-foreground">Check-in Date</p>
                  <p className="font-medium">{formatDate(booking.checkInDate)}</p>
                </div>
                <div>
                  <p className="text-sm text-muted-foreground">Check-out Date</p>
                  <p className="font-medium">{formatDate(booking.checkOutDate)}</p>
                </div>
              </div>
            </div>
          </div>

          <Separator />

          {/* Payment Information */}
          <div className="space-y-3">
            <h3 className="font-semibold text-lg flex items-center gap-2">
              <IndianRupee className="h-5 w-5 text-primary" />
              Payment Information
            </h3>
            <div className="bg-muted/50 rounded-lg p-4 space-y-3">
              <div className="flex items-center justify-between">
                <span className="text-muted-foreground">Total Price</span>
                <span className="font-semibold text-lg">{formatCurrency(booking.totalPrice)}</span>
              </div>
              <Separator />
              <div className="flex items-center justify-between">
                <span className="text-muted-foreground">Commission (10%)</span>
                <span className="font-semibold text-lg text-blue-600">{formatCurrency(booking.commission)}</span>
              </div>
              <Separator />
              <div className="flex items-center justify-between">
                <span className="font-medium">Net Amount to Hotel</span>
                <span className="font-bold text-xl text-green-600">
                  {formatCurrency(booking.totalPrice - booking.commission)}
                </span>
              </div>
            </div>
          </div>

          {/* Booking Timeline */}
          <div className="bg-muted/30 rounded-lg p-4">
            <div className="grid grid-cols-2 gap-4 text-sm">
              <div>
                <p className="text-muted-foreground">Created On</p>
                <p className="font-medium">{formatDate(booking.createdAt)}</p>
              </div>
              <div>
                <p className="text-muted-foreground">Last Updated</p>
                <p className="font-medium">{formatDate(booking.updatedAt)}</p>
              </div>
            </div>
          </div>
        </div>
      </DialogContent>
    </Dialog>
  )
}
