"use client"

import { Dialog, DialogContent, DialogDescription, DialogHeader, DialogTitle } from "@/components/ui/dialog"
import { Badge } from "@/components/ui/badge"
import { Separator } from "@/components/ui/separator"
import { ScrollArea } from "@/components/ui/scroll-area"
import { FileText, Calendar } from "lucide-react"
import type { HotelPartner } from "@/lib/data/hotel-partners"

interface ViewDocumentsDialogProps {
  partner: HotelPartner | null
  open: boolean
  onOpenChange: (open: boolean) => void
}

export function ViewDocumentsDialog({ partner, open, onOpenChange }: ViewDocumentsDialogProps) {
  if (!partner) return null

  return (
    <Dialog open={open} onOpenChange={onOpenChange}>
      <DialogContent className="max-w-lg max-h-[90vh] flex flex-col">
        <DialogHeader className="flex-shrink-0">
          <DialogTitle className="flex items-center gap-2">
            <FileText className="h-5 w-5 text-primary" aria-hidden="true" />
            {partner.hotelName} - Documents
          </DialogTitle>
          <DialogDescription>View all documents and certifications for this hotel partner.</DialogDescription>
        </DialogHeader>

        <ScrollArea className="flex-1 overflow-auto pr-4">
          <div className="space-y-6">
            {/* Partner Since */}
            <div className="flex items-center gap-3 p-3 bg-muted/50 rounded-lg">
              <Calendar className="h-5 w-5 text-primary" aria-hidden="true" />
              <div>
                <p className="text-sm text-muted-foreground">Partner Since</p>
                <p className="font-medium">
                  {new Date(partner.since).toLocaleDateString("en-IN", {
                    year: "numeric",
                    month: "long",
                    day: "numeric",
                  })}
                </p>
              </div>
            </div>

            <Separator />

            {/* Tax Documents */}
            <section aria-labelledby="tax-documents-heading">
              <h3 id="tax-documents-heading" className="text-sm font-semibold text-muted-foreground mb-3">
                Tax Documents
              </h3>
              <div className="grid gap-3">
                <div className="flex justify-between items-center p-3 bg-background border rounded-lg">
                  <span className="text-sm text-muted-foreground">GST Number</span>
                  <Badge variant="outline" className="font-mono">
                    {partner.documents.gstNumber}
                  </Badge>
                </div>
                <div className="flex justify-between items-center p-3 bg-background border rounded-lg">
                  <span className="text-sm text-muted-foreground">PAN Number</span>
                  <Badge variant="outline" className="font-mono">
                    {partner.documents.panNumber}
                  </Badge>
                </div>
              </div>
            </section>

            <Separator />

            {/* Registration Documents */}
            <section aria-labelledby="registration-documents-heading">
              <h3 id="registration-documents-heading" className="text-sm font-semibold text-muted-foreground mb-3">
                Registration Documents
              </h3>
              <div className="grid gap-3">
                <div className="flex justify-between items-center p-3 bg-background border rounded-lg">
                  <span className="text-sm text-muted-foreground">Registration No.</span>
                  <Badge variant="outline" className="font-mono">
                    {partner.documents.registrationNumber}
                  </Badge>
                </div>
                <div className="flex justify-between items-center p-3 bg-background border rounded-lg">
                  <span className="text-sm text-muted-foreground">License No.</span>
                  <Badge variant="outline" className="font-mono">
                    {partner.documents.licenseNumber}
                  </Badge>
                </div>
              </div>
            </section>

            {partner.documents.otherDocuments.length > 0 && (
              <>
                <Separator />

                {/* Other Documents */}
                <section aria-labelledby="other-documents-heading">
                  <h3 id="other-documents-heading" className="text-sm font-semibold text-muted-foreground mb-3">
                    Other Documents & Certifications
                  </h3>
                  <div className="flex flex-wrap gap-2">
                    {partner.documents.otherDocuments.map((doc, index) => (
                      <Badge key={index} variant="secondary" className="text-xs">
                        {doc}
                      </Badge>
                    ))}
                  </div>
                </section>
              </>
            )}
          </div>
        </ScrollArea>
      </DialogContent>
    </Dialog>
  )
}
