"use client"

import { useState, useMemo } from "react"
import { useRouter } from "next/navigation"
import { Button } from "@/components/ui/button"
import { Input } from "@/components/ui/input"
import { Tabs, TabsList, TabsTrigger } from "@/components/ui/tabs"
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select"
import { Plus, Search } from "lucide-react"
import { spiritualPackageDataStore } from "@/lib/data/spiritual-packages"
import { SpiritualPackagesTable } from "./spiritual-packages-table"
import { DeletePackageDialog } from "./delete-package-dialog"

export function SpiritualPackagesClient() {
  const router = useRouter()
  const [activeTab, setActiveTab] = useState<"active" | "inactive">("active")
  const [searchQuery, setSearchQuery] = useState("")
  const [sortBy, setSortBy] = useState<"name" | "locations" | "days">("name")
  const [currentPage, setCurrentPage] = useState(1)
  const [itemsPerPage, setItemsPerPage] = useState(10)
  const [deleteDialogOpen, setDeleteDialogOpen] = useState(false)
  const [packageToDelete, setPackageToDelete] = useState<{ id: string; name: string } | null>(null)

  const allPackages = spiritualPackageDataStore.getAllPackages()
  const activePackages = spiritualPackageDataStore.getActivePackages()
  const inactivePackages = spiritualPackageDataStore.getInactivePackages()

  const currentPackages = activeTab === "active" ? activePackages : inactivePackages

  const filteredPackages = useMemo(() => {
    const filtered = currentPackages.filter((pkg) => pkg.cityName.toLowerCase().includes(searchQuery.toLowerCase()))

    filtered.sort((a, b) => {
      switch (sortBy) {
        case "name":
          return a.cityName.localeCompare(b.cityName)
        case "locations":
          return b.totalLocations - a.totalLocations
        case "days":
          return b.dayPlans.length - a.dayPlans.length
        default:
          return 0
      }
    })

    return filtered
  }, [currentPackages, searchQuery, sortBy])

  const totalPages = Math.ceil(filteredPackages.length / itemsPerPage)
  const paginatedPackages = filteredPackages.slice((currentPage - 1) * itemsPerPage, currentPage * itemsPerPage)

  const handleToggleStatus = (id: string) => {
    spiritualPackageDataStore.togglePackageStatus(id)
    setCurrentPage(1)
  }

  const handleDeleteClick = (id: string, name: string) => {
    setPackageToDelete({ id, name })
    setDeleteDialogOpen(true)
  }

  const handleDeleteConfirm = () => {
    if (packageToDelete) {
      spiritualPackageDataStore.deletePackage(packageToDelete.id)
      setDeleteDialogOpen(false)
      setPackageToDelete(null)
      setCurrentPage(1)
    }
  }

  const handleEdit = (id: string) => {
    router.push(`/admin/spiritual-packages/edit/${id}`)
  }

  return (
    <div className="space-y-6">
      <div className="flex flex-col gap-4 sm:flex-row sm:items-center sm:justify-between">
        <Tabs value={activeTab} onValueChange={(value) => setActiveTab(value as "active" | "inactive")}>
          <TabsList>
            <TabsTrigger value="active">Active ({activePackages.length})</TabsTrigger>
            <TabsTrigger value="inactive">Inactive ({inactivePackages.length})</TabsTrigger>
          </TabsList>
        </Tabs>
        <Button onClick={() => router.push("/admin/spiritual-packages/add-new")}>
          <Plus className="mr-2 h-4 w-4" />
          Add New Package
        </Button>
      </div>

      <div className="flex flex-col gap-4 sm:flex-row sm:items-center">
        <div className="relative flex-1">
          <Search className="absolute left-3 top-1/2 h-4 w-4 -translate-y-1/2 text-muted-foreground" />
          <Input
            placeholder="Search by city name..."
            value={searchQuery}
            onChange={(e) => {
              setSearchQuery(e.target.value)
              setCurrentPage(1)
            }}
            className="pl-9"
          />
        </div>
        <Select
          value={sortBy}
          onValueChange={(value: any) => {
            setSortBy(value)
            setCurrentPage(1)
          }}
        >
          <SelectTrigger className="w-full sm:w-48">
            <SelectValue placeholder="Sort by" />
          </SelectTrigger>
          <SelectContent>
            <SelectItem value="name">Sort by Name</SelectItem>
            <SelectItem value="locations">Sort by Locations</SelectItem>
            <SelectItem value="days">Sort by Days</SelectItem>
          </SelectContent>
        </Select>
      </div>

      <SpiritualPackagesTable
        packages={paginatedPackages}
        onToggleStatus={handleToggleStatus}
        onEdit={handleEdit}
        onDelete={handleDeleteClick}
        currentPage={currentPage}
        totalPages={totalPages}
        itemsPerPage={itemsPerPage}
        onPageChange={setCurrentPage}
        onItemsPerPageChange={(value) => {
          setItemsPerPage(value)
          setCurrentPage(1)
        }}
      />

      <DeletePackageDialog
        open={deleteDialogOpen}
        onOpenChange={setDeleteDialogOpen}
        packageName={packageToDelete?.name || ""}
        onConfirm={handleDeleteConfirm}
      />
    </div>
  )
}
