"use client"

import { Dialog, DialogContent, DialogDescription, DialogHeader, DialogTitle } from "@/components/ui/dialog"
import { Badge } from "@/components/ui/badge"
import { Separator } from "@/components/ui/separator"
import { ScrollArea } from "@/components/ui/scroll-area"
import { User, Building2, Wallet, Calendar, FileText } from "lucide-react"
import type { WithdrawRequest } from "@/lib/data/withdrawals"

interface WithdrawDetailsDialogProps {
  withdrawal: WithdrawRequest | null
  open: boolean
  onOpenChange: (open: boolean) => void
}

function formatCurrency(amount: number): string {
  return new Intl.NumberFormat("en-IN", {
    style: "currency",
    currency: "INR",
    maximumFractionDigits: 0,
  }).format(amount)
}

function formatDate(dateString: string): string {
  return new Intl.DateTimeFormat("en-IN", {
    day: "2-digit",
    month: "long",
    year: "numeric",
    hour: "2-digit",
    minute: "2-digit",
  }).format(new Date(dateString))
}

export function WithdrawDetailsDialog({ withdrawal, open, onOpenChange }: WithdrawDetailsDialogProps) {
  if (!withdrawal) return null

  const statusVariant = {
    pending: "secondary",
    approved: "default",
    cancelled: "destructive",
  } as const

  return (
    <Dialog open={open} onOpenChange={onOpenChange}>
      <DialogContent className="sm:max-w-[550px] max-h-[90vh] flex flex-col">
        <DialogHeader className="flex-shrink-0">
          <DialogTitle className="flex items-center gap-2">
            <FileText className="h-5 w-5 text-primary" aria-hidden="true" />
            Withdrawal Details
          </DialogTitle>
          <DialogDescription>Complete information about this withdrawal request</DialogDescription>
        </DialogHeader>

        <ScrollArea className="flex-1 overflow-auto pr-4">
          <div className="space-y-6 pb-2">
            {/* Status Badge */}
            <div className="flex items-center justify-between">
              <span className="text-sm text-muted-foreground">Status</span>
              <Badge variant={statusVariant[withdrawal.status]} className="capitalize">
                {withdrawal.status}
              </Badge>
            </div>

            <Separator />

            {/* User Information */}
            <div className="space-y-3">
              <h4 className="font-medium text-sm flex items-center gap-2">
                <User className="h-4 w-4" aria-hidden="true" />
                Requester Information
              </h4>
              <div className="grid grid-cols-2 gap-4 text-sm">
                <div>
                  <span className="text-muted-foreground">Name</span>
                  <p className="font-medium">{withdrawal.name}</p>
                </div>
                <div>
                  <span className="text-muted-foreground">Type</span>
                  <p className="font-medium capitalize">{withdrawal.userType}</p>
                </div>
                <div>
                  <span className="text-muted-foreground">Email</span>
                  <p className="font-medium break-all">{withdrawal.email}</p>
                </div>
                <div>
                  <span className="text-muted-foreground">Phone</span>
                  <p className="font-medium">{withdrawal.phone}</p>
                </div>
              </div>
            </div>

            <Separator />

            {/* Financial Information */}
            <div className="space-y-3">
              <h4 className="font-medium text-sm flex items-center gap-2">
                <Wallet className="h-4 w-4" aria-hidden="true" />
                Financial Details
              </h4>
              <div className="grid grid-cols-2 gap-4 text-sm">
                <div>
                  <span className="text-muted-foreground">Wallet Balance</span>
                  <p className="font-medium text-green-600">{formatCurrency(withdrawal.walletBalance)}</p>
                </div>
                <div>
                  <span className="text-muted-foreground">Withdraw Amount</span>
                  <p className="font-medium text-primary">{formatCurrency(withdrawal.withdrawAmount)}</p>
                </div>
              </div>
            </div>

            <Separator />

            {/* Bank Details */}
            <div className="space-y-3">
              <h4 className="font-medium text-sm flex items-center gap-2">
                <Building2 className="h-4 w-4" aria-hidden="true" />
                Bank Details
              </h4>
              <div className="grid grid-cols-2 gap-4 text-sm">
                <div>
                  <span className="text-muted-foreground">Account Holder</span>
                  <p className="font-medium">{withdrawal.bankDetails.accountHolder}</p>
                </div>
                <div>
                  <span className="text-muted-foreground">Bank Name</span>
                  <p className="font-medium">{withdrawal.bankDetails.bankName}</p>
                </div>
                <div>
                  <span className="text-muted-foreground">Account Number</span>
                  <p className="font-medium font-mono">{withdrawal.bankDetails.accountNumber}</p>
                </div>
                <div>
                  <span className="text-muted-foreground">IFSC Code</span>
                  <p className="font-medium font-mono">{withdrawal.bankDetails.ifscCode}</p>
                </div>
              </div>
            </div>

            <Separator />

            {/* Timeline */}
            <div className="space-y-3">
              <h4 className="font-medium text-sm flex items-center gap-2">
                <Calendar className="h-4 w-4" aria-hidden="true" />
                Timeline
              </h4>
              <div className="grid grid-cols-2 gap-4 text-sm">
                <div>
                  <span className="text-muted-foreground">Requested At</span>
                  <p className="font-medium">{formatDate(withdrawal.requestedAt)}</p>
                </div>
                {withdrawal.processedAt && (
                  <div>
                    <span className="text-muted-foreground">Processed At</span>
                    <p className="font-medium">{formatDate(withdrawal.processedAt)}</p>
                  </div>
                )}
              </div>
            </div>

            {/* Reference Number for Approved */}
            {withdrawal.status === "approved" && withdrawal.referenceNumber && (
              <>
                <Separator />
                <div className="space-y-2">
                  <span className="text-sm text-muted-foreground">Transaction Reference</span>
                  <code className="block rounded bg-muted px-3 py-2 text-sm font-mono">
                    {withdrawal.referenceNumber}
                  </code>
                </div>
              </>
            )}

            {/* Cancel Reason for Cancelled */}
            {withdrawal.status === "cancelled" && withdrawal.cancelReason && (
              <>
                <Separator />
                <div className="space-y-2">
                  <span className="text-sm text-muted-foreground">Cancellation Reason</span>
                  <p className="text-sm rounded bg-destructive/10 text-destructive px-3 py-2">
                    {withdrawal.cancelReason}
                  </p>
                </div>
              </>
            )}
          </div>
        </ScrollArea>
      </DialogContent>
    </Dialog>
  )
}
