"use client"

import { Button } from "@/components/ui/button"
import {
  DropdownMenu,
  DropdownMenuContent,
  DropdownMenuItem,
  DropdownMenuLabel,
  DropdownMenuSeparator,
  DropdownMenuTrigger,
} from "@/components/ui/dropdown-menu"
import { User, LogOut, LayoutDashboard, Plus, List, Wallet, Settings, UserCog } from "lucide-react"
import Link from "next/link"
import { usePathname, useRouter } from "next/navigation"
import { cn } from "@/lib/utils"

interface AgentHeaderProps {
  agentEmail?: string
  agentName?: string
}

const navItems = [
  { label: "Dashboard", href: "/agent/dashboard", icon: LayoutDashboard },
  { label: "Book New Service", href: "/agent/book-new-service", icon: Plus },
  { label: "All Bookings", href: "/agent/all-bookings", icon: List },
  { label: "Earning & Withdraw", href: "/agent/earning-withdraw", icon: Wallet },
]

export function AgentHeader({ agentEmail = "agent@teertham.com", agentName = "Agent" }: AgentHeaderProps) {
  const router = useRouter()
  const pathname = usePathname()

  const handleLogout = () => {
    if (typeof window !== "undefined") {
      sessionStorage.removeItem("agent_session")
      document.cookie = "agent_session=; path=/; max-age=0; SameSite=Strict"
    }
    router.push("/login")
  }

  return (
    <header className="sticky top-0 z-50 w-full border-b border-border/40 bg-card shadow-md">
      <div className="flex h-16 items-center px-4 md:px-6 lg:px-8">
        {/* Logo */}
        <Link href="/agent/dashboard" className="flex items-center gap-3 shrink-0">
          <div className="flex h-10 w-10 items-center justify-center rounded-xl bg-gradient-to-br from-gray-800 to-gray-900 text-white shadow-lg">
            <span className="text-lg font-bold">T</span>
          </div>
          <div className="flex flex-col">
            <span className="text-xl font-semibold tracking-tight">Teertham</span>
            <span className="text-[9px] font-bold uppercase tracking-widest bg-gradient-to-r from-gray-800/20 to-gray-900/10 text-gray-700 px-2 py-0.5 rounded-md border border-gray-500/20 w-fit">
              Agent
            </span>
          </div>
        </Link>

        <nav className="hidden md:flex items-center justify-center flex-1 gap-1">
          {navItems.map((item) => {
            const isActive = pathname === item.href || pathname.startsWith(item.href + "/")
            return (
              <Link
                key={item.href}
                href={item.href}
                className={cn(
                  "flex items-center gap-2 px-4 py-2 rounded-lg text-sm font-medium transition-all",
                  isActive
                    ? "agent-btn text-white shadow-md"
                    : "text-muted-foreground hover:text-foreground hover:bg-muted",
                )}
              >
                <item.icon className="h-4 w-4" />
                {item.label}
              </Link>
            )
          })}
        </nav>

        {/* Profile dropdown */}
        <div className="flex items-center gap-3 shrink-0">
          <DropdownMenu>
            <DropdownMenuTrigger asChild>
              <Button
                variant="ghost"
                size="icon"
                className="rounded-xl ring-2 ring-border hover:ring-gray-500/30 transition-all h-10 w-10"
                aria-label="User menu"
              >
                <div className="h-8 w-8 rounded-lg bg-gradient-to-br from-gray-500/20 to-gray-500/5 flex items-center justify-center">
                  <User className="h-4 w-4 text-gray-600" aria-hidden="true" />
                </div>
              </Button>
            </DropdownMenuTrigger>
            <DropdownMenuContent align="end" className="w-60 p-2">
              <DropdownMenuLabel className="font-normal p-3 bg-muted/50 rounded-lg mb-2">
                <div className="flex flex-col space-y-1">
                  <p className="text-sm font-semibold leading-none">{agentName}</p>
                  <p className="text-xs leading-none text-muted-foreground">{agentEmail}</p>
                </div>
              </DropdownMenuLabel>
              <DropdownMenuSeparator className="my-2" />
              <DropdownMenuItem asChild className="rounded-lg py-2.5 cursor-pointer">
                <Link href="/agent/itinerary-setting" className="flex items-center">
                  <Settings className="mr-3 h-4 w-4" aria-hidden="true" />
                  Itinerary Setting
                </Link>
              </DropdownMenuItem>
              <DropdownMenuItem asChild className="rounded-lg py-2.5 cursor-pointer">
                <Link href="/agent/manage-profile" className="flex items-center">
                  <UserCog className="mr-3 h-4 w-4" aria-hidden="true" />
                  Manage Profile
                </Link>
              </DropdownMenuItem>
              <DropdownMenuSeparator className="my-2" />
              <DropdownMenuItem
                onClick={handleLogout}
                className="rounded-lg py-2.5 cursor-pointer text-destructive focus:text-destructive focus:bg-destructive/10"
              >
                <LogOut className="mr-3 h-4 w-4" aria-hidden="true" />
                Logout
              </DropdownMenuItem>
            </DropdownMenuContent>
          </DropdownMenu>
        </div>
      </div>

      {/* Mobile navigation */}
      <div className="md:hidden border-t border-border/40 px-4 py-2 overflow-x-auto">
        <nav className="flex items-center gap-2">
          {navItems.map((item) => {
            const isActive = pathname === item.href || pathname.startsWith(item.href + "/")
            return (
              <Link
                key={item.href}
                href={item.href}
                className={cn(
                  "flex items-center gap-2 px-3 py-1.5 rounded-lg text-xs font-medium whitespace-nowrap transition-all",
                  isActive
                    ? "agent-btn text-white shadow-md"
                    : "text-muted-foreground hover:text-foreground hover:bg-muted",
                )}
              >
                <item.icon className="h-3.5 w-3.5" />
                {item.label}
              </Link>
            )
          })}
        </nav>
      </div>
    </header>
  )
}
