"use client"

import type React from "react"

import { Shield, Clock, Globe, Headphones, CheckCircle2, Star, Users, TrendingUp } from "lucide-react"

interface AuthLayoutProps {
  children: React.ReactNode
  variant: "login" | "create-account" | "forget-password" | "otp-verification" | "reset-password"
}

const contentMap = {
  login: {
    title: "Welcome Back to Teertham",
    subtitle: "Your trusted pilgrimage travel partner",
    features: [
      { icon: Shield, text: "Secure & encrypted login" },
      { icon: Clock, text: "24/7 account access" },
      { icon: Globe, text: "Manage bookings anywhere" },
      { icon: Headphones, text: "Priority customer support" },
    ],
    stats: [
      { value: "50K+", label: "Happy Travelers" },
      { value: "100+", label: "Destinations" },
      { value: "4.9", label: "User Rating" },
    ],
  },
  "create-account": {
    title: "Start Your Sacred Journey",
    subtitle: "Join thousands of pilgrims who trust Teertham",
    features: [
      { icon: CheckCircle2, text: "Easy booking management" },
      { icon: Star, text: "Exclusive member discounts" },
      { icon: Users, text: "Join our travel community" },
      { icon: TrendingUp, text: "Earn rewards on every trip" },
    ],
    stats: [
      { value: "50K+", label: "Happy Travelers" },
      { value: "100+", label: "Destinations" },
      { value: "4.9", label: "User Rating" },
    ],
  },
  "forget-password": {
    title: "Don't Worry, We've Got You",
    subtitle: "Recover your account in simple steps",
    features: [
      { icon: Shield, text: "Secure password recovery" },
      { icon: Clock, text: "Quick verification process" },
      { icon: CheckCircle2, text: "Instant email delivery" },
      { icon: Headphones, text: "Support available 24/7" },
    ],
    stats: [
      { value: "2 Min", label: "Average Recovery" },
      { value: "100%", label: "Secure Process" },
      { value: "24/7", label: "Support" },
    ],
  },
  "otp-verification": {
    title: "Almost There!",
    subtitle: "Just one more step to secure your account",
    features: [
      { icon: Shield, text: "One-time secure code" },
      { icon: Clock, text: "Valid for 10 minutes" },
      { icon: CheckCircle2, text: "Check spam folder too" },
      { icon: Headphones, text: "Resend if not received" },
    ],
    stats: [
      { value: "6 Digit", label: "Secure Code" },
      { value: "10 Min", label: "Validity" },
      { value: "Instant", label: "Delivery" },
    ],
  },
  "reset-password": {
    title: "Create New Password",
    subtitle: "Make it strong and memorable",
    features: [
      { icon: Shield, text: "Use 8+ characters" },
      { icon: CheckCircle2, text: "Mix letters & numbers" },
      { icon: Star, text: "Add special characters" },
      { icon: Clock, text: "Don't reuse old passwords" },
    ],
    stats: [
      { value: "Strong", label: "Password Required" },
      { value: "Secure", label: "Encryption" },
      { value: "Instant", label: "Update" },
    ],
  },
}

export function AuthLayout({ children, variant }: AuthLayoutProps) {
  const content = contentMap[variant]

  return (
    <main className="min-h-[calc(100vh-4rem)] flex">
      <aside className="hidden lg:flex lg:w-1/2 xl:w-[45%] frontend-gradient-auth text-white relative overflow-hidden">
        {/* Background pattern - updated to teal tones */}
        <div className="absolute inset-0 opacity-10">
          <div className="absolute top-20 left-10 w-72 h-72 bg-white rounded-full blur-3xl" />
          <div className="absolute bottom-20 right-10 w-96 h-96 bg-teal-300 rounded-full blur-3xl" />
          <div className="absolute top-1/2 left-1/2 -translate-x-1/2 -translate-y-1/2 w-[500px] h-[500px] bg-white/20 rounded-full blur-3xl" />
        </div>

        <div className="relative z-10 flex flex-col justify-center p-8 lg:p-12 xl:p-16 w-full">
          <div className="space-y-8">
            {/* Logo */}
            <div className="flex items-center gap-3">
              <div className="h-12 w-12 rounded-xl bg-white/20 backdrop-blur-sm flex items-center justify-center">
                <span className="text-2xl font-bold">T</span>
              </div>
              <span className="text-2xl font-bold">Teertham</span>
            </div>

            {/* Main content */}
            <div className="space-y-4">
              <h2 className="text-3xl xl:text-4xl font-bold leading-tight text-balance">{content.title}</h2>
              <p className="text-lg text-white/80 text-pretty">{content.subtitle}</p>
            </div>

            {/* Features list */}
            <ul className="space-y-4" role="list">
              {content.features.map((feature, index) => (
                <li key={index} className="flex items-center gap-4">
                  <div className="h-10 w-10 rounded-lg bg-white/20 backdrop-blur-sm flex items-center justify-center flex-shrink-0">
                    <feature.icon className="h-5 w-5" aria-hidden="true" />
                  </div>
                  <span className="text-white/90">{feature.text}</span>
                </li>
              ))}
            </ul>

            {/* Stats */}
            <div className="flex gap-8 pt-8 border-t border-white/20">
              {content.stats.map((stat, index) => (
                <div key={index} className="text-center">
                  <p className="text-2xl xl:text-3xl font-bold">{stat.value}</p>
                  <p className="text-sm text-white/70">{stat.label}</p>
                </div>
              ))}
            </div>

            {/* Trust badge */}
            <div className="flex items-center gap-2 pt-4">
              <div className="flex -space-x-2">
                {[1, 2, 3, 4].map((i) => (
                  <div
                    key={i}
                    className="h-8 w-8 rounded-full bg-white/30 border-2 border-white/50 flex items-center justify-center text-xs font-medium"
                  >
                    {String.fromCharCode(64 + i)}
                  </div>
                ))}
              </div>
              <p className="text-sm text-white/80">
                Join <span className="font-semibold">50,000+</span> happy travelers
              </p>
            </div>
          </div>
        </div>
      </aside>

      {/* Right side - Form content */}
      <section className="flex-1 flex items-center justify-center p-4 md:p-8 bg-gradient-to-br from-background via-background to-muted/30">
        <div className="w-full max-w-md">{children}</div>
      </section>
    </main>
  )
}
