"use client"

import { useState } from "react"
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card"
import { Badge } from "@/components/ui/badge"
import { Button } from "@/components/ui/button"
import { Input } from "@/components/ui/input"
import { Label } from "@/components/ui/label"
import { Textarea } from "@/components/ui/textarea"
import { Avatar, AvatarFallback } from "@/components/ui/avatar"
import {
  Dialog,
  DialogContent,
  DialogHeader,
  DialogTitle,
  DialogDescription,
  DialogFooter,
} from "@/components/ui/dialog"
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select"
import {
  Search,
  Filter,
  MoreHorizontal,
  Eye,
  Phone,
  CalendarDays,
  Users,
  Mail,
  MapPin,
  LogIn,
  LogOut,
  Clock,
  XCircle,
  FileCheck,
  Upload,
  CreditCard,
  AlertCircle,
} from "lucide-react"
import { DropdownMenu, DropdownMenuContent, DropdownMenuItem, DropdownMenuTrigger } from "@/components/ui/dropdown-menu"

interface BookingsTableProps {
  status: string
}

const bookingsData = {
  booked: [
    {
      id: "BK001",
      guest: "Rajesh Kumar",
      email: "rajesh.kumar@email.com",
      phone: "+91 98765 43210",
      room: "Deluxe Suite - 101",
      checkIn: "Dec 15, 2025",
      checkOut: "Dec 18, 2025",
      guests: 2,
      amount: "₹12,500",
      address: "Mumbai, Maharashtra",
      idType: "Aadhaar Card",
      idNumber: "XXXX-XXXX-4521",
    },
    {
      id: "BK002",
      guest: "Priya Sharma",
      email: "priya.sharma@email.com",
      phone: "+91 87654 32109",
      room: "Standard Room - 205",
      checkIn: "Dec 16, 2025",
      checkOut: "Dec 17, 2025",
      guests: 1,
      amount: "₹4,200",
      address: "Delhi, NCR",
      idType: "Passport",
      idNumber: "N1234567",
    },
    {
      id: "BK003",
      guest: "Amit Verma",
      email: "amit.verma@email.com",
      phone: "+91 76543 21098",
      room: "Premium Suite - 301",
      checkIn: "Dec 17, 2025",
      checkOut: "Dec 20, 2025",
      guests: 3,
      amount: "₹18,000",
      address: "Bangalore, Karnataka",
      idType: "Voter ID",
      idNumber: "ABC1234567",
    },
  ],
  "checked-in": [
    {
      id: "BK010",
      guest: "Sneha Reddy",
      email: "sneha.reddy@email.com",
      phone: "+91 65432 10987",
      room: "Deluxe Room - 102",
      checkIn: "Dec 09, 2025",
      checkOut: "Dec 12, 2025",
      guests: 2,
      amount: "₹8,400",
      address: "Hyderabad, Telangana",
      idType: "Aadhaar Card",
      idNumber: "XXXX-XXXX-7832",
      checkedInAt: "Dec 09, 2025 - 2:30 PM",
    },
    {
      id: "BK011",
      guest: "Vikram Singh",
      email: "vikram.singh@email.com",
      phone: "+91 54321 09876",
      room: "Suite - 401",
      checkIn: "Dec 08, 2025",
      checkOut: "Dec 11, 2025",
      guests: 4,
      amount: "₹22,000",
      address: "Jaipur, Rajasthan",
      idType: "Driving License",
      idNumber: "RJ-01-2023-0045621",
      checkedInAt: "Dec 08, 2025 - 3:15 PM",
    },
  ],
  extended: [
    {
      id: "BK020",
      guest: "Meera Iyer",
      email: "meera.iyer@email.com",
      phone: "+91 43210 98765",
      room: "Standard Room - 108",
      checkIn: "Dec 05, 2025",
      checkOut: "Dec 12, 2025",
      guests: 1,
      amount: "₹9,800",
      extended: "+3 days",
      originalCheckout: "Dec 09, 2025",
      address: "Chennai, Tamil Nadu",
      idType: "Aadhaar Card",
      idNumber: "XXXX-XXXX-9876",
    },
  ],
  "checked-out": [
    {
      id: "BK030",
      guest: "Rahul Gupta",
      email: "rahul.gupta@email.com",
      phone: "+91 32109 87654",
      room: "Deluxe Suite - 103",
      checkIn: "Dec 01, 2025",
      checkOut: "Dec 05, 2025",
      guests: 2,
      amount: "₹16,000",
      address: "Pune, Maharashtra",
      checkedOutAt: "Dec 05, 2025 - 10:45 AM",
    },
    {
      id: "BK031",
      guest: "Anita Desai",
      email: "anita.desai@email.com",
      phone: "+91 21098 76543",
      room: "Premium Room - 202",
      checkIn: "Dec 02, 2025",
      checkOut: "Dec 04, 2025",
      guests: 2,
      amount: "₹7,600",
      address: "Ahmedabad, Gujarat",
      checkedOutAt: "Dec 04, 2025 - 11:00 AM",
    },
  ],
  cancelled: [
    {
      id: "BK040",
      guest: "Karan Malhotra",
      email: "karan.malhotra@email.com",
      phone: "+91 10987 65432",
      room: "Suite - 402",
      checkIn: "Dec 10, 2025",
      checkOut: "Dec 15, 2025",
      guests: 3,
      amount: "₹25,000",
      reason: "Personal emergency",
      address: "Kolkata, West Bengal",
      cancelledAt: "Dec 08, 2025",
      refundStatus: "Processed",
      refundAmount: "₹20,000",
    },
  ],
}

const statusStyles = {
  booked: {
    badgeClass: "bg-blue-50 text-blue-600 border-blue-200 dark:bg-blue-950 dark:text-blue-400 dark:border-blue-800",
    label: "Booked",
  },
  "checked-in": {
    badgeClass:
      "bg-emerald-50 text-emerald-600 border-emerald-200 dark:bg-emerald-950 dark:text-emerald-400 dark:border-emerald-800",
    label: "Checked In",
  },
  extended: {
    badgeClass:
      "bg-violet-50 text-violet-600 border-violet-200 dark:bg-violet-950 dark:text-violet-400 dark:border-violet-800",
    label: "Extended",
  },
  "checked-out": {
    badgeClass:
      "bg-slate-50 text-slate-600 border-slate-200 dark:bg-slate-900 dark:text-slate-400 dark:border-slate-700",
    label: "Checked Out",
  },
  cancelled: {
    badgeClass: "bg-red-50 text-red-600 border-red-200 dark:bg-red-950 dark:text-red-400 dark:border-red-800",
    label: "Cancelled",
  },
}

export function BookingsTable({ status }: BookingsTableProps) {
  const [searchQuery, setSearchQuery] = useState("")
  const [selectedBooking, setSelectedBooking] = useState<(typeof bookingsData.booked)[0] | null>(null)
  const [dialogType, setDialogType] = useState<
    "view" | "checkin" | "checkout" | "extend" | "cancel" | "payment" | null
  >(null)
  const [extendDays, setExtendDays] = useState("1")
  const [cancelReason, setCancelReason] = useState("")
  const [idProofFile, setIdProofFile] = useState<File | null>(null)

  const bookings = bookingsData[status as keyof typeof bookingsData] || []
  const statusStyle = statusStyles[status as keyof typeof statusStyles]

  const filteredBookings = bookings.filter(
    (booking) =>
      booking.guest.toLowerCase().includes(searchQuery.toLowerCase()) ||
      booking.id.toLowerCase().includes(searchQuery.toLowerCase()) ||
      booking.room.toLowerCase().includes(searchQuery.toLowerCase()),
  )

  const openDialog = (booking: (typeof bookingsData.booked)[0], type: typeof dialogType) => {
    setSelectedBooking(booking)
    setDialogType(type)
  }

  const closeDialog = () => {
    setSelectedBooking(null)
    setDialogType(null)
    setExtendDays("1")
    setCancelReason("")
    setIdProofFile(null)
  }

  const handleAction = () => {
    // Handle the action based on dialogType
    closeDialog()
  }

  return (
    <>
      <Card className="border bg-card min-h-[400px]">
        <CardHeader>
          <div className="flex flex-col sm:flex-row sm:items-center sm:justify-between gap-4">
            <div>
              <CardTitle className="text-xl font-semibold flex items-center gap-3">
                {statusStyle.label} Bookings
                <Badge variant="outline" className={statusStyle.badgeClass}>
                  {filteredBookings.length}
                </Badge>
              </CardTitle>
            </div>

            <div className="flex items-center gap-2">
              <div className="relative">
                <Search className="absolute left-3 top-1/2 -translate-y-1/2 h-4 w-4 text-muted-foreground" />
                <Input
                  placeholder="Search bookings..."
                  value={searchQuery}
                  onChange={(e) => setSearchQuery(e.target.value)}
                  className="pl-9 w-[200px]"
                />
              </div>
              <Button variant="outline" size="icon">
                <Filter className="h-4 w-4" />
              </Button>
            </div>
          </div>
        </CardHeader>

        <CardContent>
          {filteredBookings.length === 0 ? (
            <div className="text-center py-12">
              <CalendarDays className="h-12 w-12 mx-auto text-muted-foreground/50 mb-3" />
              <p className="text-muted-foreground">No bookings found</p>
            </div>
          ) : (
            <div className="space-y-3">
              {filteredBookings.map((booking) => (
                <div
                  key={booking.id}
                  className="group flex flex-col gap-4 p-4 rounded-xl bg-muted/30 hover:bg-muted/50 border border-border/50 transition-all duration-200"
                >
                  <div className="flex flex-col sm:flex-row sm:items-center gap-4">
                    <Avatar className="h-12 w-12 border shrink-0">
                      <AvatarFallback className="bg-slate-100 dark:bg-slate-800 text-slate-600 dark:text-slate-300 font-semibold">
                        {booking.guest
                          .split(" ")
                          .map((n) => n[0])
                          .join("")}
                      </AvatarFallback>
                    </Avatar>

                    <div className="flex-1 min-w-0 grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-4 gap-3">
                      <div>
                        <p className="font-semibold truncate">{booking.guest}</p>
                        <p className="text-xs text-muted-foreground flex items-center gap-1">
                          <Mail className="h-3 w-3" />
                          {booking.email}
                        </p>
                        <p className="text-xs text-muted-foreground flex items-center gap-1">
                          <Phone className="h-3 w-3" />
                          {booking.phone}
                        </p>
                      </div>

                      <div>
                        <p className="font-medium text-sm">{booking.room}</p>
                        <p className="text-xs text-muted-foreground flex items-center gap-1">
                          <Users className="h-3 w-3" />
                          {booking.guests} Guest{booking.guests > 1 ? "s" : ""}
                        </p>
                        <p className="text-xs text-muted-foreground flex items-center gap-1">
                          <MapPin className="h-3 w-3" />
                          {booking.address}
                        </p>
                      </div>

                      <div>
                        <p className="text-sm flex items-center gap-1">
                          <CalendarDays className="h-3 w-3 text-muted-foreground" />
                          {booking.checkIn} - {booking.checkOut}
                        </p>
                        {"extended" in booking && (
                          <Badge
                            variant="outline"
                            className="bg-violet-50 text-violet-600 border-violet-200 dark:bg-violet-950 dark:text-violet-400 text-xs mt-1"
                          >
                            {booking.extended}
                          </Badge>
                        )}
                        {"reason" in booking && <p className="text-xs text-red-500 mt-1">{booking.reason}</p>}
                        {"checkedInAt" in booking && (
                          <p className="text-xs text-emerald-600 dark:text-emerald-400 mt-1">
                            Checked in: {booking.checkedInAt}
                          </p>
                        )}
                        {"checkedOutAt" in booking && (
                          <p className="text-xs text-slate-500 mt-1">Checked out: {booking.checkedOutAt}</p>
                        )}
                      </div>

                      <div className="flex items-center justify-between sm:justify-end gap-3">
                        <p className="font-bold text-lg">{booking.amount}</p>

                        <DropdownMenu>
                          <DropdownMenuTrigger asChild>
                            <Button variant="ghost" size="icon" className="h-8 w-8">
                              <MoreHorizontal className="h-4 w-4" />
                            </Button>
                          </DropdownMenuTrigger>
                          <DropdownMenuContent align="end">
                            <DropdownMenuItem onClick={() => openDialog(booking, "view")}>
                              <Eye className="h-4 w-4 mr-2" />
                              View Details
                            </DropdownMenuItem>
                            <DropdownMenuItem>
                              <Phone className="h-4 w-4 mr-2" />
                              Contact Guest
                            </DropdownMenuItem>
                            <DropdownMenuItem>
                              <Mail className="h-4 w-4 mr-2" />
                              Send Email
                            </DropdownMenuItem>
                          </DropdownMenuContent>
                        </DropdownMenu>
                      </div>
                    </div>
                  </div>

                  {/* Action Buttons based on status */}
                  <div className="flex flex-wrap items-center gap-2 pt-2 border-t border-border/50">
                    {status === "booked" && (
                      <>
                        <Button
                          size="sm"
                          className="bg-emerald-600 hover:bg-emerald-700 text-white"
                          onClick={() => openDialog(booking, "checkin")}
                        >
                          <LogIn className="h-3.5 w-3.5 mr-1.5" />
                          Check-In
                        </Button>
                        <Button size="sm" variant="outline" onClick={() => openDialog(booking, "payment")}>
                          <CreditCard className="h-3.5 w-3.5 mr-1.5" />
                          Record Payment
                        </Button>
                        <Button
                          size="sm"
                          variant="outline"
                          className="text-red-600 hover:text-red-700 hover:bg-red-50 bg-transparent"
                          onClick={() => openDialog(booking, "cancel")}
                        >
                          <XCircle className="h-3.5 w-3.5 mr-1.5" />
                          Cancel Booking
                        </Button>
                      </>
                    )}

                    {status === "checked-in" && (
                      <>
                        <Button
                          size="sm"
                          className="bg-slate-700 hover:bg-slate-800 text-white"
                          onClick={() => openDialog(booking, "checkout")}
                        >
                          <LogOut className="h-3.5 w-3.5 mr-1.5" />
                          Check-Out
                        </Button>
                        <Button
                          size="sm"
                          variant="outline"
                          className="text-violet-600 hover:text-violet-700 hover:bg-violet-50 bg-transparent"
                          onClick={() => openDialog(booking, "extend")}
                        >
                          <Clock className="h-3.5 w-3.5 mr-1.5" />
                          Extend Stay
                        </Button>
                        <Button size="sm" variant="outline" onClick={() => openDialog(booking, "payment")}>
                          <CreditCard className="h-3.5 w-3.5 mr-1.5" />
                          Add Payment
                        </Button>
                      </>
                    )}

                    {status === "extended" && (
                      <>
                        <Button
                          size="sm"
                          className="bg-slate-700 hover:bg-slate-800 text-white"
                          onClick={() => openDialog(booking, "checkout")}
                        >
                          <LogOut className="h-3.5 w-3.5 mr-1.5" />
                          Check-Out
                        </Button>
                        <Button
                          size="sm"
                          variant="outline"
                          className="text-violet-600 hover:text-violet-700 hover:bg-violet-50 bg-transparent"
                          onClick={() => openDialog(booking, "extend")}
                        >
                          <Clock className="h-3.5 w-3.5 mr-1.5" />
                          Extend More
                        </Button>
                      </>
                    )}

                    {status === "checked-out" && (
                      <Button size="sm" variant="outline" onClick={() => openDialog(booking, "view")}>
                        <FileCheck className="h-3.5 w-3.5 mr-1.5" />
                        View Invoice
                      </Button>
                    )}

                    {status === "cancelled" && "refundStatus" in booking && (
                      <Badge
                        variant="outline"
                        className="bg-emerald-50 text-emerald-600 border-emerald-200 dark:bg-emerald-950 dark:text-emerald-400"
                      >
                        Refund {booking.refundStatus}: {booking.refundAmount}
                      </Badge>
                    )}
                  </div>
                </div>
              ))}
            </div>
          )}
        </CardContent>
      </Card>

      {/* Check-In Dialog */}
      <Dialog open={dialogType === "checkin"} onOpenChange={(open) => !open && closeDialog()}>
        <DialogContent className="max-w-lg">
          <DialogHeader>
            <DialogTitle className="flex items-center gap-2">
              <LogIn className="h-5 w-5 text-emerald-600" />
              Check-In Guest
            </DialogTitle>
            <DialogDescription>Complete the check-in process for {selectedBooking?.guest}</DialogDescription>
          </DialogHeader>

          <div className="space-y-4 py-4">
            <div className="p-4 rounded-lg bg-muted/50 border">
              <div className="grid grid-cols-2 gap-3 text-sm">
                <div>
                  <p className="text-muted-foreground">Booking ID</p>
                  <p className="font-medium">{selectedBooking?.id}</p>
                </div>
                <div>
                  <p className="text-muted-foreground">Room</p>
                  <p className="font-medium">{selectedBooking?.room}</p>
                </div>
                <div>
                  <p className="text-muted-foreground">Check-In Date</p>
                  <p className="font-medium">{selectedBooking?.checkIn}</p>
                </div>
                <div>
                  <p className="text-muted-foreground">Check-Out Date</p>
                  <p className="font-medium">{selectedBooking?.checkOut}</p>
                </div>
              </div>
            </div>

            <div className="space-y-3">
              <Label className="text-base font-medium">ID Proof Verification</Label>

              <div className="grid grid-cols-2 gap-3">
                <div className="space-y-2">
                  <Label htmlFor="idType">ID Type</Label>
                  <Select defaultValue={selectedBooking?.idType}>
                    <SelectTrigger>
                      <SelectValue placeholder="Select ID type" />
                    </SelectTrigger>
                    <SelectContent>
                      <SelectItem value="Aadhaar Card">Aadhaar Card</SelectItem>
                      <SelectItem value="Passport">Passport</SelectItem>
                      <SelectItem value="Voter ID">Voter ID</SelectItem>
                      <SelectItem value="Driving License">Driving License</SelectItem>
                      <SelectItem value="PAN Card">PAN Card</SelectItem>
                    </SelectContent>
                  </Select>
                </div>
                <div className="space-y-2">
                  <Label htmlFor="idNumber">ID Number</Label>
                  <Input id="idNumber" defaultValue={selectedBooking?.idNumber} placeholder="Enter ID number" />
                </div>
              </div>

              <div className="space-y-2">
                <Label>Upload ID Proof Photo</Label>
                <div className="border-2 border-dashed rounded-lg p-6 text-center hover:border-slate-400 transition-colors cursor-pointer">
                  <input
                    type="file"
                    id="idProof"
                    className="hidden"
                    accept="image/*"
                    onChange={(e) => setIdProofFile(e.target.files?.[0] || null)}
                  />
                  <label htmlFor="idProof" className="cursor-pointer">
                    <Upload className="h-8 w-8 mx-auto text-muted-foreground mb-2" />
                    <p className="text-sm text-muted-foreground">
                      {idProofFile ? idProofFile.name : "Click to upload or drag and drop"}
                    </p>
                    <p className="text-xs text-muted-foreground mt-1">PNG, JPG up to 5MB</p>
                  </label>
                </div>
              </div>

              <div className="flex items-start gap-2 p-3 rounded-lg bg-blue-50 dark:bg-blue-950 border border-blue-200 dark:border-blue-800">
                <AlertCircle className="h-4 w-4 text-blue-600 dark:text-blue-400 mt-0.5" />
                <p className="text-xs text-blue-700 dark:text-blue-300">
                  Please verify the guest&apos;s identity before completing check-in. Ensure all details match the
                  booking information.
                </p>
              </div>
            </div>
          </div>

          <DialogFooter>
            <Button variant="outline" onClick={closeDialog}>
              Cancel
            </Button>
            <Button className="bg-emerald-600 hover:bg-emerald-700" onClick={handleAction}>
              <LogIn className="h-4 w-4 mr-2" />
              Complete Check-In
            </Button>
          </DialogFooter>
        </DialogContent>
      </Dialog>

      {/* Check-Out Dialog */}
      <Dialog open={dialogType === "checkout"} onOpenChange={(open) => !open && closeDialog()}>
        <DialogContent className="max-w-lg">
          <DialogHeader>
            <DialogTitle className="flex items-center gap-2">
              <LogOut className="h-5 w-5 text-slate-600" />
              Check-Out Guest
            </DialogTitle>
            <DialogDescription>Complete the check-out process for {selectedBooking?.guest}</DialogDescription>
          </DialogHeader>

          <div className="space-y-4 py-4">
            <div className="p-4 rounded-lg bg-muted/50 border">
              <div className="grid grid-cols-2 gap-3 text-sm">
                <div>
                  <p className="text-muted-foreground">Guest Name</p>
                  <p className="font-medium">{selectedBooking?.guest}</p>
                </div>
                <div>
                  <p className="text-muted-foreground">Room</p>
                  <p className="font-medium">{selectedBooking?.room}</p>
                </div>
                <div>
                  <p className="text-muted-foreground">Stay Duration</p>
                  <p className="font-medium">
                    {selectedBooking?.checkIn} - {selectedBooking?.checkOut}
                  </p>
                </div>
                <div>
                  <p className="text-muted-foreground">Total Amount</p>
                  <p className="font-bold text-lg">{selectedBooking?.amount}</p>
                </div>
              </div>
            </div>

            <div className="space-y-3">
              <div className="space-y-2">
                <Label htmlFor="additionalCharges">Additional Charges (if any)</Label>
                <Input id="additionalCharges" placeholder="₹0" />
              </div>

              <div className="space-y-2">
                <Label htmlFor="checkoutNotes">Notes</Label>
                <Textarea id="checkoutNotes" placeholder="Any notes about the check-out..." rows={3} />
              </div>

              <div className="flex items-center justify-between p-4 rounded-lg bg-slate-100 dark:bg-slate-800">
                <span className="font-medium">Final Amount</span>
                <span className="text-xl font-bold">{selectedBooking?.amount}</span>
              </div>
            </div>
          </div>

          <DialogFooter>
            <Button variant="outline" onClick={closeDialog}>
              Cancel
            </Button>
            <Button className="bg-slate-700 hover:bg-slate-800" onClick={handleAction}>
              <LogOut className="h-4 w-4 mr-2" />
              Complete Check-Out
            </Button>
          </DialogFooter>
        </DialogContent>
      </Dialog>

      {/* Extend Stay Dialog */}
      <Dialog open={dialogType === "extend"} onOpenChange={(open) => !open && closeDialog()}>
        <DialogContent className="max-w-md">
          <DialogHeader>
            <DialogTitle className="flex items-center gap-2">
              <Clock className="h-5 w-5 text-violet-600" />
              Extend Stay
            </DialogTitle>
            <DialogDescription>Extend the booking duration for {selectedBooking?.guest}</DialogDescription>
          </DialogHeader>

          <div className="space-y-4 py-4">
            <div className="p-4 rounded-lg bg-muted/50 border">
              <div className="grid grid-cols-2 gap-3 text-sm">
                <div>
                  <p className="text-muted-foreground">Current Check-Out</p>
                  <p className="font-medium">{selectedBooking?.checkOut}</p>
                </div>
                <div>
                  <p className="text-muted-foreground">Room</p>
                  <p className="font-medium">{selectedBooking?.room}</p>
                </div>
              </div>
            </div>

            <div className="space-y-2">
              <Label htmlFor="extendDays">Extend By</Label>
              <Select value={extendDays} onValueChange={setExtendDays}>
                <SelectTrigger>
                  <SelectValue placeholder="Select days" />
                </SelectTrigger>
                <SelectContent>
                  {[1, 2, 3, 4, 5, 6, 7].map((day) => (
                    <SelectItem key={day} value={day.toString()}>
                      {day} {day === 1 ? "Day" : "Days"}
                    </SelectItem>
                  ))}
                </SelectContent>
              </Select>
            </div>

            <div className="p-4 rounded-lg bg-violet-50 dark:bg-violet-950 border border-violet-200 dark:border-violet-800">
              <div className="flex items-center justify-between">
                <span className="text-sm font-medium text-violet-700 dark:text-violet-300">
                  Additional Amount ({extendDays} {Number(extendDays) === 1 ? "day" : "days"})
                </span>
                <span className="text-lg font-bold text-violet-700 dark:text-violet-300">
                  ₹{Number(extendDays) * 4200}
                </span>
              </div>
            </div>
          </div>

          <DialogFooter>
            <Button variant="outline" onClick={closeDialog}>
              Cancel
            </Button>
            <Button className="bg-violet-600 hover:bg-violet-700" onClick={handleAction}>
              <Clock className="h-4 w-4 mr-2" />
              Confirm Extension
            </Button>
          </DialogFooter>
        </DialogContent>
      </Dialog>

      {/* Cancel Booking Dialog */}
      <Dialog open={dialogType === "cancel"} onOpenChange={(open) => !open && closeDialog()}>
        <DialogContent className="max-w-md">
          <DialogHeader>
            <DialogTitle className="flex items-center gap-2 text-red-600">
              <XCircle className="h-5 w-5" />
              Cancel Booking
            </DialogTitle>
            <DialogDescription>This action will cancel the booking for {selectedBooking?.guest}</DialogDescription>
          </DialogHeader>

          <div className="space-y-4 py-4">
            <div className="p-4 rounded-lg bg-red-50 dark:bg-red-950 border border-red-200 dark:border-red-800">
              <div className="flex items-start gap-2">
                <AlertCircle className="h-4 w-4 text-red-600 dark:text-red-400 mt-0.5" />
                <div className="text-sm text-red-700 dark:text-red-300">
                  <p className="font-medium">Warning</p>
                  <p>Cancelling this booking may result in a refund being processed to the guest.</p>
                </div>
              </div>
            </div>

            <div className="space-y-2">
              <Label htmlFor="cancelReason">Reason for Cancellation</Label>
              <Select value={cancelReason} onValueChange={setCancelReason}>
                <SelectTrigger>
                  <SelectValue placeholder="Select reason" />
                </SelectTrigger>
                <SelectContent>
                  <SelectItem value="guest_request">Guest Request</SelectItem>
                  <SelectItem value="no_show">No Show</SelectItem>
                  <SelectItem value="overbooking">Overbooking</SelectItem>
                  <SelectItem value="maintenance">Room Maintenance</SelectItem>
                  <SelectItem value="other">Other</SelectItem>
                </SelectContent>
              </Select>
            </div>

            <div className="space-y-2">
              <Label htmlFor="cancelNotes">Additional Notes</Label>
              <Textarea id="cancelNotes" placeholder="Provide additional details..." rows={3} />
            </div>
          </div>

          <DialogFooter>
            <Button variant="outline" onClick={closeDialog}>
              Keep Booking
            </Button>
            <Button variant="destructive" onClick={handleAction}>
              <XCircle className="h-4 w-4 mr-2" />
              Cancel Booking
            </Button>
          </DialogFooter>
        </DialogContent>
      </Dialog>

      {/* Record Payment Dialog */}
      <Dialog open={dialogType === "payment"} onOpenChange={(open) => !open && closeDialog()}>
        <DialogContent className="max-w-md">
          <DialogHeader>
            <DialogTitle className="flex items-center gap-2">
              <CreditCard className="h-5 w-5 text-blue-600" />
              Record Payment
            </DialogTitle>
            <DialogDescription>Add payment record for {selectedBooking?.guest}</DialogDescription>
          </DialogHeader>

          <div className="space-y-4 py-4">
            <div className="space-y-2">
              <Label htmlFor="paymentAmount">Amount</Label>
              <Input id="paymentAmount" placeholder="₹0" />
            </div>

            <div className="space-y-2">
              <Label htmlFor="paymentMethod">Payment Method</Label>
              <Select>
                <SelectTrigger>
                  <SelectValue placeholder="Select method" />
                </SelectTrigger>
                <SelectContent>
                  <SelectItem value="cash">Cash</SelectItem>
                  <SelectItem value="card">Credit/Debit Card</SelectItem>
                  <SelectItem value="upi">UPI</SelectItem>
                  <SelectItem value="netbanking">Net Banking</SelectItem>
                  <SelectItem value="wallet">Digital Wallet</SelectItem>
                </SelectContent>
              </Select>
            </div>

            <div className="space-y-2">
              <Label htmlFor="transactionId">Transaction ID (Optional)</Label>
              <Input id="transactionId" placeholder="Enter transaction ID" />
            </div>

            <div className="space-y-2">
              <Label htmlFor="paymentNotes">Notes</Label>
              <Textarea id="paymentNotes" placeholder="Any additional notes..." rows={2} />
            </div>
          </div>

          <DialogFooter>
            <Button variant="outline" onClick={closeDialog}>
              Cancel
            </Button>
            <Button className="bg-blue-600 hover:bg-blue-700" onClick={handleAction}>
              <CreditCard className="h-4 w-4 mr-2" />
              Record Payment
            </Button>
          </DialogFooter>
        </DialogContent>
      </Dialog>

      {/* View Details Dialog */}
      <Dialog open={dialogType === "view"} onOpenChange={(open) => !open && closeDialog()}>
        <DialogContent className="max-w-lg">
          <DialogHeader>
            <DialogTitle>Booking Details</DialogTitle>
          </DialogHeader>

          <div className="space-y-4 py-4">
            <div className="flex items-center gap-4">
              <Avatar className="h-16 w-16 border">
                <AvatarFallback className="bg-slate-100 dark:bg-slate-800 text-slate-600 dark:text-slate-300 font-semibold text-lg">
                  {selectedBooking?.guest
                    .split(" ")
                    .map((n) => n[0])
                    .join("")}
                </AvatarFallback>
              </Avatar>
              <div>
                <h3 className="text-xl font-semibold">{selectedBooking?.guest}</h3>
                <p className="text-sm text-muted-foreground">Booking #{selectedBooking?.id}</p>
              </div>
            </div>

            <div className="grid grid-cols-2 gap-4 p-4 rounded-lg bg-muted/50 border">
              <div>
                <p className="text-xs text-muted-foreground">Email</p>
                <p className="font-medium text-sm">{selectedBooking?.email}</p>
              </div>
              <div>
                <p className="text-xs text-muted-foreground">Phone</p>
                <p className="font-medium text-sm">{selectedBooking?.phone}</p>
              </div>
              <div>
                <p className="text-xs text-muted-foreground">Room</p>
                <p className="font-medium text-sm">{selectedBooking?.room}</p>
              </div>
              <div>
                <p className="text-xs text-muted-foreground">Guests</p>
                <p className="font-medium text-sm">{selectedBooking?.guests}</p>
              </div>
              <div>
                <p className="text-xs text-muted-foreground">Check-In</p>
                <p className="font-medium text-sm">{selectedBooking?.checkIn}</p>
              </div>
              <div>
                <p className="text-xs text-muted-foreground">Check-Out</p>
                <p className="font-medium text-sm">{selectedBooking?.checkOut}</p>
              </div>
              <div>
                <p className="text-xs text-muted-foreground">Address</p>
                <p className="font-medium text-sm">{selectedBooking?.address}</p>
              </div>
              <div>
                <p className="text-xs text-muted-foreground">Total Amount</p>
                <p className="font-bold text-lg">{selectedBooking?.amount}</p>
              </div>
            </div>
          </div>

          <DialogFooter>
            <Button variant="outline" onClick={closeDialog}>
              Close
            </Button>
          </DialogFooter>
        </DialogContent>
      </Dialog>
    </>
  )
}
