"use client"

import { Button } from "@/components/ui/button"
import {
  DropdownMenu,
  DropdownMenuContent,
  DropdownMenuItem,
  DropdownMenuLabel,
  DropdownMenuSeparator,
  DropdownMenuTrigger,
} from "@/components/ui/dropdown-menu"
import { User, LogOut, Settings, KeyRound, Bell, LayoutDashboard, CalendarCheck, Settings2, Wallet } from "lucide-react"
import Link from "next/link"
import { useRouter, usePathname } from "next/navigation"
import { cn } from "@/lib/utils"

const navTabs = [
  {
    name: "Dashboard",
    href: "/hotel/dashboard",
    icon: LayoutDashboard,
  },
  {
    name: "Bookings",
    href: "/hotel/bookings",
    icon: CalendarCheck,
  },
  {
    name: "Hotel & Room Settings",
    href: "/hotel/room-settings",
    icon: Settings2,
  },
  {
    name: "Earning & Withdraw",
    href: "/hotel/withdraw",
    icon: Wallet,
  },
]

interface HotelHeaderProps {
  hotelEmail?: string
  hotelName?: string
}

export function HotelHeader({ hotelEmail = "hotel@teertham.com", hotelName = "Hotel" }: HotelHeaderProps) {
  const router = useRouter()
  const pathname = usePathname()

  const handleLogout = () => {
    if (typeof window !== "undefined") {
      sessionStorage.removeItem("hotel_session")
      document.cookie = "hotel_session=; path=/; max-age=0; SameSite=Strict"
    }
    router.push("/login")
  }

  const isActiveTab = (href: string) => {
    if (href === "/hotel/dashboard") {
      return pathname === "/hotel/dashboard"
    }
    return pathname.startsWith(href)
  }

  return (
    <header className="sticky top-0 z-50 w-full border-b border-border/40 bg-card/95 backdrop-blur-md shadow-sm">
      <div className="flex h-16 items-center justify-between px-4 md:px-6 lg:px-8">
        {/* Logo */}
        <Link href="/hotel/dashboard" className="flex items-center gap-3 shrink-0">
          <div className="flex h-10 w-10 items-center justify-center rounded-xl bg-gradient-to-br from-slate-700 to-slate-900 text-white shadow-lg">
            <span className="text-lg font-bold">T</span>
          </div>
          <div className="flex flex-col">
            <span className="text-xl font-semibold tracking-tight">Teertham</span>
            <span className="text-[9px] font-bold uppercase tracking-widest bg-slate-100 dark:bg-slate-800 text-slate-600 dark:text-slate-300 px-2 py-0.5 rounded-md border border-slate-200 dark:border-slate-700 w-fit">
              Hotel
            </span>
          </div>
        </Link>

        <nav className="hidden md:flex items-center gap-1 ml-8" aria-label="Main navigation">
          {navTabs.map((tab) => {
            const isActive = isActiveTab(tab.href)
            const Icon = tab.icon

            return (
              <Link
                key={tab.href}
                href={tab.href}
                className={cn(
                  "relative flex items-center gap-2 px-4 py-2 rounded-lg font-medium text-sm transition-all duration-200",
                  isActive
                    ? "bg-slate-900 dark:bg-white text-white dark:text-slate-900 shadow-md"
                    : "text-muted-foreground hover:text-foreground hover:bg-slate-100 dark:hover:bg-slate-800",
                )}
              >
                <Icon className="h-4 w-4" />
                <span className="hidden lg:inline">{tab.name}</span>
              </Link>
            )
          })}
        </nav>

        <div className="flex items-center gap-3">
          <Link href="/hotel/notifications">
            <Button
              variant="ghost"
              size="icon"
              className="rounded-lg h-9 w-9 text-muted-foreground hover:text-foreground"
              aria-label="Notifications"
            >
              <Bell className="h-4 w-4" />
            </Button>
          </Link>

          <DropdownMenu>
            <DropdownMenuTrigger asChild>
              <Button
                variant="ghost"
                size="icon"
                className="rounded-xl ring-2 ring-border hover:ring-slate-400 transition-all h-10 w-10"
                aria-label="User menu"
              >
                <div className="h-8 w-8 rounded-lg bg-gradient-to-br from-slate-100 to-slate-200 dark:from-slate-700 dark:to-slate-800 flex items-center justify-center">
                  <User className="h-4 w-4 text-slate-600 dark:text-slate-300" aria-hidden="true" />
                </div>
              </Button>
            </DropdownMenuTrigger>
            <DropdownMenuContent align="end" className="w-64 p-2">
              <DropdownMenuLabel className="font-normal p-3 bg-muted/50 rounded-lg mb-2">
                <div className="flex flex-col space-y-1">
                  <p className="text-sm font-semibold leading-none">{hotelName}</p>
                  <p className="text-xs leading-none text-muted-foreground">{hotelEmail}</p>
                </div>
              </DropdownMenuLabel>

              <DropdownMenuItem asChild className="rounded-lg py-2.5 cursor-pointer">
                <Link href="/hotel/manage-profile">
                  <Settings className="mr-3 h-4 w-4" aria-hidden="true" />
                  Manage Profile
                </Link>
              </DropdownMenuItem>

              <DropdownMenuItem asChild className="rounded-lg py-2.5 cursor-pointer">
                <Link href="/hotel/manage-profile?tab=security">
                  <KeyRound className="mr-3 h-4 w-4" aria-hidden="true" />
                  Change Password
                </Link>
              </DropdownMenuItem>

              <DropdownMenuSeparator className="my-2" />

              <DropdownMenuItem
                onClick={handleLogout}
                className="rounded-lg py-2.5 cursor-pointer text-destructive focus:text-destructive focus:bg-destructive/10"
              >
                <LogOut className="mr-3 h-4 w-4" aria-hidden="true" />
                Logout
              </DropdownMenuItem>
            </DropdownMenuContent>
          </DropdownMenu>
        </div>
      </div>

      {/* Mobile Navigation */}
      <div className="md:hidden border-t border-border/40 px-4 py-2 overflow-x-auto">
        <nav className="flex items-center gap-1" aria-label="Mobile navigation">
          {navTabs.map((tab) => {
            const isActive = isActiveTab(tab.href)
            const Icon = tab.icon

            return (
              <Link
                key={tab.href}
                href={tab.href}
                className={cn(
                  "flex items-center gap-2 px-3 py-2 rounded-lg font-medium text-xs whitespace-nowrap transition-all duration-200",
                  isActive
                    ? "bg-slate-900 dark:bg-white text-white dark:text-slate-900"
                    : "text-muted-foreground hover:bg-slate-100 dark:hover:bg-slate-800",
                )}
              >
                <Icon className="h-3.5 w-3.5" />
                <span>{tab.name}</span>
              </Link>
            )
          })}
        </nav>
      </div>
    </header>
  )
}
