"use client"

import { useState } from "react"
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card"
import { Button } from "@/components/ui/button"
import { Input } from "@/components/ui/input"
import { Label } from "@/components/ui/label"
import { Textarea } from "@/components/ui/textarea"
import { Switch } from "@/components/ui/switch"
import { Badge } from "@/components/ui/badge"
import {
  Building2,
  MapPin,
  Phone,
  Mail,
  Globe,
  Clock,
  Save,
  Upload,
  ImageIcon,
  Wifi,
  Car,
  UtensilsCrossed,
  Dumbbell,
  Waves,
  Wind,
  Tv,
  Coffee,
  Sparkles,
  ShieldCheck,
  Baby,
  PawPrint,
  Cigarette,
  Accessibility,
  Plane,
  Shirt,
  Briefcase,
  Flower2,
  Gamepad2,
  Music,
  BookOpen,
  HeartPulse,
  Mountain,
  TreePine,
  Bike,
} from "lucide-react"

const hotelFacilities = [
  { id: "wifi", label: "Free WiFi", icon: Wifi },
  { id: "parking", label: "Parking", icon: Car },
  { id: "restaurant", label: "Restaurant", icon: UtensilsCrossed },
  { id: "gym", label: "Gym/Fitness Center", icon: Dumbbell },
  { id: "pool", label: "Swimming Pool", icon: Waves },
  { id: "ac", label: "Air Conditioning", icon: Wind },
  { id: "tv", label: "Cable/Smart TV", icon: Tv },
  { id: "breakfast", label: "Complimentary Breakfast", icon: Coffee },
  { id: "housekeeping", label: "Daily Housekeeping", icon: Sparkles },
  { id: "security", label: "24/7 Security", icon: ShieldCheck },
  { id: "kids", label: "Kids Play Area", icon: Baby },
  { id: "pets", label: "Pet Friendly", icon: PawPrint },
  { id: "smoking", label: "Smoking Area", icon: Cigarette },
  { id: "accessible", label: "Wheelchair Accessible", icon: Accessibility },
  { id: "airport", label: "Airport Shuttle", icon: Plane },
  { id: "laundry", label: "Laundry Service", icon: Shirt },
  { id: "business", label: "Business Center", icon: Briefcase },
  { id: "spa", label: "Spa & Wellness", icon: Flower2 },
  { id: "gaming", label: "Game Room", icon: Gamepad2 },
  { id: "entertainment", label: "Live Entertainment", icon: Music },
  { id: "library", label: "Library/Reading Room", icon: BookOpen },
  { id: "medical", label: "Medical Assistance", icon: HeartPulse },
  { id: "view", label: "Mountain/Sea View", icon: Mountain },
  { id: "garden", label: "Garden Area", icon: TreePine },
  { id: "cycling", label: "Bicycle Rental", icon: Bike },
]

export function HotelSettingsForm() {
  const [selectedFacilities, setSelectedFacilities] = useState<string[]>([
    "wifi",
    "parking",
    "restaurant",
    "ac",
    "tv",
    "housekeeping",
    "security",
  ])
  const [hotelImages, setHotelImages] = useState<string[]>([
    "/placeholder.svg?height=200&width=300",
    "/placeholder.svg?height=200&width=300",
  ])

  const toggleFacility = (facilityId: string) => {
    setSelectedFacilities((prev) =>
      prev.includes(facilityId) ? prev.filter((id) => id !== facilityId) : [...prev, facilityId],
    )
  }

  const handleImageUpload = () => {
    const newImage = `/placeholder.svg?height=200&width=300&query=hotel room ${hotelImages.length + 1}`
    setHotelImages((prev) => [...prev, newImage])
  }

  const removeImage = (index: number) => {
    setHotelImages((prev) => prev.filter((_, i) => i !== index))
  }

  return (
    <div className="space-y-6">
      {/* Basic Information */}
      <Card className="border-0 shadow-sm">
        <CardHeader>
          <CardTitle className="flex items-center gap-2">
            <Building2 className="h-5 w-5 text-teal-600" />
            Basic Information
          </CardTitle>
          <CardDescription>Update your hotel's basic details</CardDescription>
        </CardHeader>
        <CardContent className="space-y-4">
          <div className="grid gap-4 md:grid-cols-2">
            <div className="space-y-2">
              <Label htmlFor="hotelName">Hotel Name</Label>
              <Input id="hotelName" defaultValue="Grand Palace Hotel" />
            </div>
            <div className="space-y-2">
              <Label htmlFor="hotelType">Hotel Type</Label>
              <select id="hotelType" className="w-full rounded-md border border-input bg-background px-3 py-2">
                <option value="hotel">Hotel</option>
                <option value="resort">Resort</option>
                <option value="villa">Villa</option>
                <option value="apartment">Apartment</option>
                <option value="hostel">Hostel</option>
                <option value="guesthouse">Guest House</option>
              </select>
            </div>
          </div>
          <div className="space-y-2">
            <Label htmlFor="description">Description</Label>
            <Textarea
              id="description"
              rows={4}
              defaultValue="Experience luxury and comfort at Grand Palace Hotel. Located in the heart of the city, we offer world-class amenities, exceptional service, and unforgettable experiences for both business and leisure travelers."
            />
          </div>
          <div className="grid gap-4 md:grid-cols-3">
            <div className="space-y-2">
              <Label htmlFor="starRating">Star Rating</Label>
              <select id="starRating" className="w-full rounded-md border border-input bg-background px-3 py-2">
                <option value="5">5 Star</option>
                <option value="4">4 Star</option>
                <option value="3">3 Star</option>
                <option value="2">2 Star</option>
                <option value="1">1 Star</option>
              </select>
            </div>
            <div className="space-y-2">
              <Label htmlFor="totalRooms">Total Rooms</Label>
              <Input id="totalRooms" type="number" defaultValue="50" />
            </div>
            <div className="space-y-2">
              <Label htmlFor="establishedYear">Established Year</Label>
              <Input id="establishedYear" type="number" defaultValue="2010" />
            </div>
          </div>
        </CardContent>
      </Card>

      {/* Hotel Photos */}
      <Card className="border-0 shadow-sm">
        <CardHeader>
          <CardTitle className="flex items-center gap-2">
            <ImageIcon className="h-5 w-5 text-teal-600" />
            Hotel Photos
          </CardTitle>
          <CardDescription>Upload photos of your hotel (exterior, lobby, facilities)</CardDescription>
        </CardHeader>
        <CardContent>
          <div className="grid gap-4 grid-cols-2 md:grid-cols-4">
            {hotelImages.map((image, index) => (
              <div key={index} className="relative group aspect-video rounded-lg overflow-hidden border">
                <img
                  src={image || "/placeholder.svg"}
                  alt={`Hotel ${index + 1}`}
                  className="w-full h-full object-cover"
                />
                <div className="absolute inset-0 bg-black/50 opacity-0 group-hover:opacity-100 transition-opacity flex items-center justify-center">
                  <Button variant="destructive" size="sm" onClick={() => removeImage(index)}>
                    Remove
                  </Button>
                </div>
              </div>
            ))}
            <button
              onClick={handleImageUpload}
              className="aspect-video rounded-lg border-2 border-dashed border-muted-foreground/25 hover:border-teal-500 hover:bg-teal-50 transition-colors flex flex-col items-center justify-center gap-2 text-muted-foreground hover:text-teal-600"
            >
              <Upload className="h-8 w-8" />
              <span className="text-sm font-medium">Add Photo</span>
            </button>
          </div>
        </CardContent>
      </Card>

      {/* Contact & Location */}
      <Card className="border-0 shadow-sm">
        <CardHeader>
          <CardTitle className="flex items-center gap-2">
            <MapPin className="h-5 w-5 text-teal-600" />
            Contact & Location
          </CardTitle>
          <CardDescription>Update your hotel's contact information and address</CardDescription>
        </CardHeader>
        <CardContent className="space-y-4">
          <div className="space-y-2">
            <Label htmlFor="address">Full Address</Label>
            <Textarea
              id="address"
              rows={2}
              defaultValue="123 Main Street, Downtown District, New Delhi, India - 110001"
            />
          </div>
          <div className="grid gap-4 md:grid-cols-3">
            <div className="space-y-2">
              <Label htmlFor="city">City</Label>
              <Input id="city" defaultValue="New Delhi" />
            </div>
            <div className="space-y-2">
              <Label htmlFor="state">State</Label>
              <Input id="state" defaultValue="Delhi" />
            </div>
            <div className="space-y-2">
              <Label htmlFor="pincode">PIN Code</Label>
              <Input id="pincode" defaultValue="110001" />
            </div>
          </div>
          <div className="grid gap-4 md:grid-cols-2">
            <div className="space-y-2">
              <Label htmlFor="phone" className="flex items-center gap-2">
                <Phone className="h-4 w-4" /> Phone Number
              </Label>
              <Input id="phone" defaultValue="+91 98765 43210" />
            </div>
            <div className="space-y-2">
              <Label htmlFor="altPhone" className="flex items-center gap-2">
                <Phone className="h-4 w-4" /> Alternate Phone
              </Label>
              <Input id="altPhone" defaultValue="+91 98765 43211" />
            </div>
          </div>
          <div className="grid gap-4 md:grid-cols-2">
            <div className="space-y-2">
              <Label htmlFor="email" className="flex items-center gap-2">
                <Mail className="h-4 w-4" /> Email Address
              </Label>
              <Input id="email" type="email" defaultValue="info@grandpalace.com" />
            </div>
            <div className="space-y-2">
              <Label htmlFor="website" className="flex items-center gap-2">
                <Globe className="h-4 w-4" /> Website
              </Label>
              <Input id="website" defaultValue="https://grandpalace.com" />
            </div>
          </div>
          <div className="grid gap-4 md:grid-cols-2">
            <div className="space-y-2">
              <Label htmlFor="latitude">Latitude</Label>
              <Input id="latitude" defaultValue="28.6139" />
            </div>
            <div className="space-y-2">
              <Label htmlFor="longitude">Longitude</Label>
              <Input id="longitude" defaultValue="77.2090" />
            </div>
          </div>
        </CardContent>
      </Card>

      {/* Facilities & Amenities */}
      <Card className="border-0 shadow-sm">
        <CardHeader>
          <CardTitle className="flex items-center gap-2">
            <Sparkles className="h-5 w-5 text-teal-600" />
            Facilities & Amenities
          </CardTitle>
          <CardDescription>Select all the facilities your hotel offers</CardDescription>
        </CardHeader>
        <CardContent>
          <div className="grid gap-3 grid-cols-2 md:grid-cols-3 lg:grid-cols-5">
            {hotelFacilities.map((facility) => {
              const Icon = facility.icon
              const isSelected = selectedFacilities.includes(facility.id)
              return (
                <button
                  key={facility.id}
                  onClick={() => toggleFacility(facility.id)}
                  className={`flex flex-col items-center gap-2 p-4 rounded-xl border-2 transition-all ${
                    isSelected
                      ? "border-teal-500 bg-teal-50 text-teal-700"
                      : "border-muted hover:border-teal-200 hover:bg-muted/50"
                  }`}
                >
                  <Icon className={`h-6 w-6 ${isSelected ? "text-teal-600" : "text-muted-foreground"}`} />
                  <span className="text-xs font-medium text-center">{facility.label}</span>
                </button>
              )
            })}
          </div>
          <div className="mt-4 flex flex-wrap gap-2">
            <span className="text-sm text-muted-foreground">Selected:</span>
            {selectedFacilities.map((id) => {
              const facility = hotelFacilities.find((f) => f.id === id)
              return facility ? (
                <Badge key={id} variant="secondary" className="bg-teal-100 text-teal-700">
                  {facility.label}
                </Badge>
              ) : null
            })}
          </div>
        </CardContent>
      </Card>

      {/* Timing & Policies */}
      <Card className="border-0 shadow-sm">
        <CardHeader>
          <CardTitle className="flex items-center gap-2">
            <Clock className="h-5 w-5 text-teal-600" />
            Timing & Policies
          </CardTitle>
          <CardDescription>Set check-in/out times and hotel policies</CardDescription>
        </CardHeader>
        <CardContent className="space-y-4">
          <div className="grid gap-4 md:grid-cols-2">
            <div className="space-y-2">
              <Label htmlFor="checkIn">Check-in Time</Label>
              <Input id="checkIn" type="time" defaultValue="14:00" />
            </div>
            <div className="space-y-2">
              <Label htmlFor="checkOut">Check-out Time</Label>
              <Input id="checkOut" type="time" defaultValue="11:00" />
            </div>
          </div>
          <div className="space-y-2">
            <Label htmlFor="cancellation">Cancellation Policy</Label>
            <Textarea
              id="cancellation"
              rows={3}
              defaultValue="Free cancellation up to 24 hours before check-in. Cancellations made within 24 hours of check-in will be charged for the first night. No-shows will be charged the full booking amount."
            />
          </div>
          <div className="space-y-2">
            <Label htmlFor="terms">Terms & Conditions</Label>
            <Textarea
              id="terms"
              rows={3}
              defaultValue="Valid government ID required at check-in. Guests must be 18+ to check-in. Damage to hotel property will be charged. Outside food and beverages are not allowed in rooms."
            />
          </div>
          <div className="grid gap-4 md:grid-cols-2">
            <div className="flex items-center justify-between p-4 border rounded-lg">
              <div>
                <p className="font-medium">Accept Couples</p>
                <p className="text-sm text-muted-foreground">Allow unmarried couples</p>
              </div>
              <Switch defaultChecked />
            </div>
            <div className="flex items-center justify-between p-4 border rounded-lg">
              <div>
                <p className="font-medium">Instant Booking</p>
                <p className="text-sm text-muted-foreground">Auto-confirm bookings</p>
              </div>
              <Switch defaultChecked />
            </div>
          </div>
        </CardContent>
      </Card>

      <div className="flex justify-end">
        <Button className="bg-teal-600 hover:bg-teal-700">
          <Save className="mr-2 h-4 w-4" />
          Save Hotel Settings
        </Button>
      </div>
    </div>
  )
}
