"use client"

import { useState } from "react"
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card"
import { Badge } from "@/components/ui/badge"
import { Button } from "@/components/ui/button"
import { Input } from "@/components/ui/input"
import { Search, Filter, Wallet, Calendar, CheckCircle, XCircle, Clock, Building } from "lucide-react"

interface WithdrawTableProps {
  status: string
}

const withdrawData = {
  pending: [
    {
      id: "WD001",
      amount: "₹45,000",
      requestDate: "Dec 08, 2025",
      bankAccount: "HDFC Bank ****4521",
      bankName: "HDFC Bank",
      expectedDate: "Dec 12, 2025",
      method: "NEFT",
    },
    {
      id: "WD002",
      amount: "₹28,500",
      requestDate: "Dec 09, 2025",
      bankAccount: "ICICI Bank ****7832",
      bankName: "ICICI Bank",
      expectedDate: "Dec 13, 2025",
      method: "IMPS",
    },
  ],
  approved: [
    {
      id: "WD010",
      amount: "₹62,000",
      requestDate: "Dec 01, 2025",
      bankAccount: "HDFC Bank ****4521",
      bankName: "HDFC Bank",
      processedDate: "Dec 03, 2025",
      transactionId: "TXN123456789",
      method: "NEFT",
    },
    {
      id: "WD011",
      amount: "₹35,800",
      requestDate: "Dec 02, 2025",
      bankAccount: "ICICI Bank ****7832",
      bankName: "ICICI Bank",
      processedDate: "Dec 04, 2025",
      transactionId: "TXN123457890",
      method: "IMPS",
    },
    {
      id: "WD012",
      amount: "₹89,200",
      requestDate: "Nov 28, 2025",
      bankAccount: "HDFC Bank ****4521",
      bankName: "HDFC Bank",
      processedDate: "Nov 30, 2025",
      transactionId: "TXN123458901",
      method: "NEFT",
    },
  ],
  cancelled: [
    {
      id: "WD020",
      amount: "₹15,000",
      requestDate: "Nov 25, 2025",
      bankAccount: "HDFC Bank ****4521",
      bankName: "HDFC Bank",
      cancelledDate: "Nov 26, 2025",
      reason: "Insufficient earnings balance",
    },
  ],
}

const statusConfig = {
  pending: {
    icon: Clock,
    badgeClass:
      "bg-amber-50 text-amber-600 border-amber-200 dark:bg-amber-950 dark:text-amber-400 dark:border-amber-800",
    label: "Pending",
  },
  approved: {
    icon: CheckCircle,
    badgeClass:
      "bg-emerald-50 text-emerald-600 border-emerald-200 dark:bg-emerald-950 dark:text-emerald-400 dark:border-emerald-800",
    label: "Approved",
  },
  cancelled: {
    icon: XCircle,
    badgeClass: "bg-red-50 text-red-600 border-red-200 dark:bg-red-950 dark:text-red-400 dark:border-red-800",
    label: "Cancelled",
  },
}

export function WithdrawTable({ status }: WithdrawTableProps) {
  const [searchQuery, setSearchQuery] = useState("")
  const withdrawals = withdrawData[status as keyof typeof withdrawData] || []
  const config = statusConfig[status as keyof typeof statusConfig]
  const StatusIcon = config?.icon || Clock

  const filteredWithdrawals = withdrawals.filter(
    (w) => w.id.toLowerCase().includes(searchQuery.toLowerCase()) || w.amount.includes(searchQuery),
  )

  return (
    <Card className="border bg-card">
      <CardHeader>
        <div className="flex flex-col sm:flex-row sm:items-center sm:justify-between gap-4">
          <CardTitle className="text-xl font-semibold flex items-center gap-3">
            {config?.label} Withdrawals
            <Badge variant="outline" className={config?.badgeClass}>
              {filteredWithdrawals.length}
            </Badge>
          </CardTitle>

          <div className="flex items-center gap-2">
            <div className="relative">
              <Search className="absolute left-3 top-1/2 -translate-y-1/2 h-4 w-4 text-muted-foreground" />
              <Input
                placeholder="Search..."
                value={searchQuery}
                onChange={(e) => setSearchQuery(e.target.value)}
                className="pl-9 w-[180px]"
              />
            </div>
            <Button variant="outline" size="icon">
              <Filter className="h-4 w-4" />
            </Button>
          </div>
        </div>
      </CardHeader>

      <CardContent>
        {filteredWithdrawals.length === 0 ? (
          <div className="text-center py-12">
            <Wallet className="h-12 w-12 mx-auto text-muted-foreground/50 mb-3" />
            <p className="text-muted-foreground">No withdrawals found</p>
          </div>
        ) : (
          <div className="space-y-3">
            {filteredWithdrawals.map((withdrawal) => (
              <div
                key={withdrawal.id}
                className="flex flex-col sm:flex-row sm:items-center gap-4 p-4 rounded-xl bg-muted/30 hover:bg-muted/50 border border-border/50 transition-all duration-200"
              >
                <div className="p-2.5 rounded-xl bg-slate-100 dark:bg-slate-800 shrink-0">
                  <StatusIcon className="h-5 w-5 text-slate-600 dark:text-slate-300" />
                </div>

                <div className="flex-1 grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-4 gap-3">
                  <div>
                    <p className="text-2xl font-bold">{withdrawal.amount}</p>
                    <p className="text-xs text-muted-foreground">#{withdrawal.id}</p>
                  </div>

                  <div>
                    <div className="flex items-center gap-1.5 text-sm font-medium">
                      <Building className="h-3.5 w-3.5 text-muted-foreground" />
                      {withdrawal.bankName}
                    </div>
                    <p className="text-xs text-muted-foreground">{withdrawal.bankAccount}</p>
                    {"method" in withdrawal && (
                      <Badge variant="secondary" className="mt-1 text-xs">
                        {withdrawal.method}
                      </Badge>
                    )}
                  </div>

                  <div>
                    <p className="text-xs text-muted-foreground flex items-center gap-1">
                      <Calendar className="h-3 w-3" />
                      Requested: {withdrawal.requestDate}
                    </p>
                    {status === "pending" && "expectedDate" in withdrawal && (
                      <p className="text-xs text-muted-foreground mt-1">Expected: {withdrawal.expectedDate}</p>
                    )}
                    {status === "approved" && "processedDate" in withdrawal && (
                      <>
                        <p className="text-xs text-emerald-600 dark:text-emerald-400 mt-1">
                          Processed: {withdrawal.processedDate}
                        </p>
                        {"transactionId" in withdrawal && (
                          <p className="text-xs text-muted-foreground">TXN: {withdrawal.transactionId}</p>
                        )}
                      </>
                    )}
                    {status === "cancelled" && "cancelledDate" in withdrawal && (
                      <>
                        <p className="text-xs text-red-600 dark:text-red-400 mt-1">
                          Cancelled: {withdrawal.cancelledDate}
                        </p>
                        {"reason" in withdrawal && <p className="text-xs text-muted-foreground">{withdrawal.reason}</p>}
                      </>
                    )}
                  </div>

                  <div className="flex items-center justify-end">
                    <Badge variant="outline" className={config?.badgeClass}>
                      {config?.label}
                    </Badge>
                  </div>
                </div>
              </div>
            ))}
          </div>
        )}
      </CardContent>
    </Card>
  )
}
