export type RouteStatus = "active" | "inactive"

export interface RouteVehicle {
  id: string
  vehicleId: string
  vehicleName: string
  vehicleCategory: string
  vehiclePhoto: string
  basePerKmRate: number // Vehicle's default per km rate
  routePerKmRate: number // Route-specific per km rate (can be different)
  addedAt: string
}

export interface Route {
  id: string
  origin: string // City name
  originCityId: string
  destination: string // City name
  destinationCityId: string
  vehicles: RouteVehicle[]
  status: RouteStatus
  createdAt: string
  updatedAt: string
}

// In-memory route data store
class RouteDataStore {
  private routes: Map<string, Route> = new Map()
  private routeIdCounter = 0
  private routeVehicleIdCounter = 0

  constructor() {
    this.initializeSampleData()
  }

  private initializeSampleData() {
    // Sample routes with vehicles
    const sampleRoutes = [
      {
        origin: "Varanasi",
        originCityId: "city_001",
        destination: "Prayagraj",
        destinationCityId: "city_004",
        status: "active" as RouteStatus,
        vehicles: [
          {
            vehicleId: "vehicle_001",
            vehicleName: "Honda City",
            vehicleCategory: "Sedan",
            vehiclePhoto: "/placeholder.svg?height=80&width=120",
            basePerKmRate: 12,
            routePerKmRate: 14,
          },
          {
            vehicleId: "vehicle_004",
            vehicleName: "Toyota Fortuner",
            vehicleCategory: "SUV",
            vehiclePhoto: "/placeholder.svg?height=80&width=120",
            basePerKmRate: 20,
            routePerKmRate: 22,
          },
          {
            vehicleId: "vehicle_010",
            vehicleName: "Toyota Innova",
            vehicleCategory: "MUV",
            vehiclePhoto: "/placeholder.svg?height=80&width=120",
            basePerKmRate: 16,
            routePerKmRate: 18,
          },
        ],
      },
      {
        origin: "Haridwar",
        originCityId: "city_006",
        destination: "Rishikesh",
        destinationCityId: "city_007",
        status: "active" as RouteStatus,
        vehicles: [
          {
            vehicleId: "vehicle_002",
            vehicleName: "Maruti Dzire",
            vehicleCategory: "Sedan",
            vehiclePhoto: "/placeholder.svg?height=80&width=120",
            basePerKmRate: 10,
            routePerKmRate: 11,
          },
          {
            vehicleId: "vehicle_007",
            vehicleName: "Maruti Swift",
            vehicleCategory: "Hatchback",
            vehiclePhoto: "/placeholder.svg?height=80&width=120",
            basePerKmRate: 8,
            routePerKmRate: 9,
          },
        ],
      },
      {
        origin: "Jaipur",
        originCityId: "city_011",
        destination: "Pushkar",
        destinationCityId: "city_012",
        status: "active" as RouteStatus,
        vehicles: [
          {
            vehicleId: "vehicle_005",
            vehicleName: "Mahindra Scorpio",
            vehicleCategory: "SUV",
            vehiclePhoto: "/placeholder.svg?height=80&width=120",
            basePerKmRate: 15,
            routePerKmRate: 16,
          },
          {
            vehicleId: "vehicle_011",
            vehicleName: "Maruti Ertiga",
            vehicleCategory: "MUV",
            vehiclePhoto: "/placeholder.svg?height=80&width=120",
            basePerKmRate: 12,
            routePerKmRate: 13,
          },
          {
            vehicleId: "vehicle_012",
            vehicleName: "Force Tempo Traveller 12 Seater",
            vehicleCategory: "Tempo Traveller",
            vehiclePhoto: "/placeholder.svg?height=80&width=120",
            basePerKmRate: 22,
            routePerKmRate: 24,
          },
        ],
      },
      {
        origin: "Mumbai",
        originCityId: "city_019",
        destination: "Shirdi",
        destinationCityId: "city_016",
        status: "active" as RouteStatus,
        vehicles: [
          {
            vehicleId: "vehicle_003",
            vehicleName: "Hyundai Verna",
            vehicleCategory: "Sedan",
            vehiclePhoto: "/placeholder.svg?height=80&width=120",
            basePerKmRate: 13,
            routePerKmRate: 15,
          },
          {
            vehicleId: "vehicle_004",
            vehicleName: "Toyota Fortuner",
            vehicleCategory: "SUV",
            vehiclePhoto: "/placeholder.svg?height=80&width=120",
            basePerKmRate: 20,
            routePerKmRate: 24,
          },
        ],
      },
      {
        origin: "Chennai",
        originCityId: "city_024",
        destination: "Rameswaram",
        destinationCityId: "city_021",
        status: "inactive" as RouteStatus,
        vehicles: [
          {
            vehicleId: "vehicle_010",
            vehicleName: "Toyota Innova",
            vehicleCategory: "MUV",
            vehiclePhoto: "/placeholder.svg?height=80&width=120",
            basePerKmRate: 16,
            routePerKmRate: 18,
          },
        ],
      },
      {
        origin: "Bangalore",
        originCityId: "city_029",
        destination: "Mysore",
        destinationCityId: "city_028",
        status: "inactive" as RouteStatus,
        vehicles: [
          {
            vehicleId: "vehicle_008",
            vehicleName: "Hyundai i20",
            vehicleCategory: "Hatchback",
            vehiclePhoto: "/placeholder.svg?height=80&width=120",
            basePerKmRate: 9,
            routePerKmRate: 10,
          },
        ],
      },
    ]

    sampleRoutes.forEach((route) => {
      this.addRoute({
        origin: route.origin,
        originCityId: route.originCityId,
        destination: route.destination,
        destinationCityId: route.destinationCityId,
        status: route.status,
        vehicles: route.vehicles.map((v) => ({
          vehicleId: v.vehicleId,
          vehicleName: v.vehicleName,
          vehicleCategory: v.vehicleCategory,
          vehiclePhoto: v.vehiclePhoto,
          basePerKmRate: v.basePerKmRate,
          routePerKmRate: v.routePerKmRate,
        })),
      })
    })
  }

  // Route methods
  getAllRoutes(): Route[] {
    return Array.from(this.routes.values()).sort((a, b) => {
      const originCompare = a.origin.localeCompare(b.origin)
      return originCompare !== 0 ? originCompare : a.destination.localeCompare(b.destination)
    })
  }

  getActiveRoutes(): Route[] {
    return this.getAllRoutes().filter((r) => r.status === "active")
  }

  getInactiveRoutes(): Route[] {
    return this.getAllRoutes().filter((r) => r.status === "inactive")
  }

  getRouteById(id: string): Route | undefined {
    return this.routes.get(id)
  }

  getRouteByOriginDestination(originCityId: string, destinationCityId: string): Route | undefined {
    return Array.from(this.routes.values()).find(
      (r) => r.originCityId === originCityId && r.destinationCityId === destinationCityId,
    )
  }

  addRoute(data: {
    origin: string
    originCityId: string
    destination: string
    destinationCityId: string
    status?: RouteStatus
    vehicles?: Omit<RouteVehicle, "id" | "addedAt">[]
  }): Route | null {
    // Check for duplicate route
    if (this.getRouteByOriginDestination(data.originCityId, data.destinationCityId)) {
      return null
    }

    this.routeIdCounter++
    const id = `route_${String(this.routeIdCounter).padStart(3, "0")}`
    const now = new Date().toISOString().split("T")[0]

    const vehicles: RouteVehicle[] = (data.vehicles || []).map((v) => {
      this.routeVehicleIdCounter++
      return {
        id: `route_vehicle_${String(this.routeVehicleIdCounter).padStart(3, "0")}`,
        ...v,
        addedAt: now,
      }
    })

    const route: Route = {
      id,
      origin: data.origin.trim(),
      originCityId: data.originCityId,
      destination: data.destination.trim(),
      destinationCityId: data.destinationCityId,
      vehicles,
      status: data.status || "active",
      createdAt: now,
      updatedAt: now,
    }

    this.routes.set(id, route)
    return route
  }

  updateRoute(
    id: string,
    data: {
      origin: string
      originCityId: string
      destination: string
      destinationCityId: string
      vehicles?: Omit<RouteVehicle, "id" | "addedAt">[]
    },
  ): Route | null {
    const route = this.routes.get(id)
    if (!route) return null

    // Check for duplicate (excluding current)
    const existing = this.getRouteByOriginDestination(data.originCityId, data.destinationCityId)
    if (existing && existing.id !== id) {
      return null
    }

    const now = new Date().toISOString().split("T")[0]

    const vehicles: RouteVehicle[] = (data.vehicles || []).map((v) => {
      this.routeVehicleIdCounter++
      return {
        id: `route_vehicle_${String(this.routeVehicleIdCounter).padStart(3, "0")}`,
        ...v,
        addedAt: now,
      }
    })

    const updatedRoute: Route = {
      ...route,
      origin: data.origin.trim(),
      originCityId: data.originCityId,
      destination: data.destination.trim(),
      destinationCityId: data.destinationCityId,
      vehicles,
      updatedAt: now,
    }

    this.routes.set(id, updatedRoute)
    return updatedRoute
  }

  addVehicleToRoute(routeId: string, vehicle: Omit<RouteVehicle, "id" | "addedAt">): Route | null {
    const route = this.routes.get(routeId)
    if (!route) return null

    // Check if vehicle already exists in route
    if (route.vehicles.some((v) => v.vehicleId === vehicle.vehicleId)) {
      return null
    }

    this.routeVehicleIdCounter++
    const newVehicle: RouteVehicle = {
      id: `route_vehicle_${String(this.routeVehicleIdCounter).padStart(3, "0")}`,
      ...vehicle,
      addedAt: new Date().toISOString().split("T")[0],
    }

    const updatedRoute: Route = {
      ...route,
      vehicles: [...route.vehicles, newVehicle],
      updatedAt: new Date().toISOString().split("T")[0],
    }

    this.routes.set(routeId, updatedRoute)
    return updatedRoute
  }

  updateRouteVehicle(routeId: string, routeVehicleId: string, routePerKmRate: number): Route | null {
    const route = this.routes.get(routeId)
    if (!route) return null

    const updatedVehicles = route.vehicles.map((v) => (v.id === routeVehicleId ? { ...v, routePerKmRate } : v))

    const updatedRoute: Route = {
      ...route,
      vehicles: updatedVehicles,
      updatedAt: new Date().toISOString().split("T")[0],
    }

    this.routes.set(routeId, updatedRoute)
    return updatedRoute
  }

  removeVehicleFromRoute(routeId: string, routeVehicleId: string): Route | null {
    const route = this.routes.get(routeId)
    if (!route) return null

    const updatedRoute: Route = {
      ...route,
      vehicles: route.vehicles.filter((v) => v.id !== routeVehicleId),
      updatedAt: new Date().toISOString().split("T")[0],
    }

    this.routes.set(routeId, updatedRoute)
    return updatedRoute
  }

  toggleRouteStatus(id: string): Route | null {
    const route = this.routes.get(id)
    if (!route) return null

    const updatedRoute: Route = {
      ...route,
      status: route.status === "active" ? "inactive" : "active",
      updatedAt: new Date().toISOString().split("T")[0],
    }

    this.routes.set(id, updatedRoute)
    return updatedRoute
  }

  deleteRoute(id: string): boolean {
    return this.routes.delete(id)
  }
}

export const routeDataStore = new RouteDataStore()
