export type PackageStatus = "active" | "inactive"

export interface PackageLocation {
  id: string
  name: string
  image: string
  itineraryDescription: string
  otherDetails: string
}

export interface PackageVehicle {
  id: string
  vehicleId: string
  vehicleName: string
  vehiclePhoto: string
  perDayPrice: number
}

export interface PackageHotel {
  id: string
  hotelName: string
  hotelPhoto: string
}

export interface PackagePricing {
  b2cWeekdayPerHead: number
  b2cWeekendPerHead: number
  b2bAgentPricing: number
}

export interface DayPlan {
  dayNumber: number
  locations: PackageLocation[]
  vehicles: PackageVehicle[]
  hotels: PackageHotel[]
  pricing: PackagePricing
}

export interface SpiritualPackage {
  id: string
  cityName: string
  totalLocations: number
  dayPlans: DayPlan[]
  status: PackageStatus
  createdAt: string
  updatedAt: string
}

class SpiritualPackageDataStore {
  private packages: Map<string, SpiritualPackage> = new Map()
  private packageIdCounter = 0

  constructor() {
    this.initializeSampleData()
  }

  private initializeSampleData() {
    const samplePackages: Omit<SpiritualPackage, "id">[] = [
      {
        cityName: "Varanasi",
        totalLocations: 5,
        dayPlans: [
          {
            dayNumber: 1,
            locations: [
              {
                id: "loc_001",
                name: "Kashi Vishwanath Temple",
                image: "/placeholder.svg?height=200&width=300",
                itineraryDescription: "Visit the famous Kashi Vishwanath Temple, one of the twelve Jyotirlingas",
                otherDetails: "Best time to visit: Early morning. Dress modestly.",
              },
              {
                id: "loc_002",
                name: "Dashashwamedh Ghat",
                image: "/placeholder.svg?height=200&width=300",
                itineraryDescription: "Experience the evening Ganga Aarti at Dashashwamedh Ghat",
                otherDetails: "Aarti timing: 7:00 PM daily",
              },
            ],
            vehicles: [
              {
                id: "veh_001",
                vehicleId: "vehicle_002",
                vehicleName: "Maruti Dzire",
                vehiclePhoto: "/placeholder.svg?height=80&width=120",
                perDayPrice: 2000,
              },
            ],
            hotels: [
              {
                id: "hotel_001",
                hotelName: "Hotel Ganges View",
                hotelPhoto: "/placeholder.svg?height=80&width=120",
              },
            ],
            pricing: {
              b2cWeekdayPerHead: 1500,
              b2cWeekendPerHead: 1800,
              b2bAgentPricing: 1200,
            },
          },
          {
            dayNumber: 2,
            locations: [
              {
                id: "loc_003",
                name: "Sarnath",
                image: "/placeholder.svg?height=200&width=300",
                itineraryDescription: "Visit Sarnath where Buddha gave his first sermon",
                otherDetails: "Museum available. Entry fee applicable.",
              },
            ],
            vehicles: [
              {
                id: "veh_002",
                vehicleId: "vehicle_002",
                vehicleName: "Maruti Dzire",
                vehiclePhoto: "/placeholder.svg?height=80&width=120",
                perDayPrice: 2000,
              },
            ],
            hotels: [
              {
                id: "hotel_002",
                hotelName: "Sarnath Residency",
                hotelPhoto: "/placeholder.svg?height=80&width=120",
              },
            ],
            pricing: {
              b2cWeekdayPerHead: 1200,
              b2cWeekendPerHead: 1400,
              b2bAgentPricing: 1000,
            },
          },
        ],
        status: "active",
        createdAt: "2024-01-15",
        updatedAt: "2024-01-15",
      },
      {
        cityName: "Ayodhya",
        totalLocations: 3,
        dayPlans: [
          {
            dayNumber: 1,
            locations: [
              {
                id: "loc_004",
                name: "Ram Mandir",
                image: "/placeholder.svg?height=200&width=300",
                itineraryDescription: "Visit the newly constructed Ram Mandir",
                otherDetails: "Online booking recommended",
              },
              {
                id: "loc_005",
                name: "Hanuman Garhi",
                image: "/placeholder.svg?height=200&width=300",
                itineraryDescription: "Visit the historic Hanuman Garhi temple",
                otherDetails: "76 stairs to climb",
              },
            ],
            vehicles: [
              {
                id: "veh_003",
                vehicleId: "vehicle_009",
                vehicleName: "Toyota Innova",
                vehiclePhoto: "/placeholder.svg?height=80&width=120",
                perDayPrice: 4000,
              },
            ],
            hotels: [
              {
                id: "hotel_003",
                hotelName: "Ayodhya Palace Hotel",
                hotelPhoto: "/placeholder.svg?height=80&width=120",
              },
            ],
            pricing: {
              b2cWeekdayPerHead: 2000,
              b2cWeekendPerHead: 2500,
              b2bAgentPricing: 1800,
            },
          },
        ],
        status: "active",
        createdAt: "2024-01-16",
        updatedAt: "2024-01-16",
      },
      {
        cityName: "Haridwar",
        totalLocations: 4,
        dayPlans: [
          {
            dayNumber: 1,
            locations: [
              {
                id: "loc_006",
                name: "Har Ki Pauri",
                image: "/placeholder.svg?height=200&width=300",
                itineraryDescription: "Attend the evening Ganga Aarti at Har Ki Pauri",
                otherDetails: "Aarti timing: 6:30 PM",
              },
            ],
            vehicles: [
              {
                id: "veh_004",
                vehicleId: "vehicle_001",
                vehicleName: "Honda City",
                vehiclePhoto: "/placeholder.svg?height=80&width=120",
                perDayPrice: 2500,
              },
            ],
            hotels: [
              {
                id: "hotel_004",
                hotelName: "Ganga Kinare Hotel",
                hotelPhoto: "/placeholder.svg?height=80&width=120",
              },
            ],
            pricing: {
              b2cWeekdayPerHead: 1300,
              b2cWeekendPerHead: 1600,
              b2bAgentPricing: 1100,
            },
          },
        ],
        status: "active",
        createdAt: "2024-01-17",
        updatedAt: "2024-01-17",
      },
    ]

    samplePackages.forEach((pkg) => {
      this.packageIdCounter++
      const id = `spiritual_pkg_${String(this.packageIdCounter).padStart(3, "0")}`
      this.packages.set(id, { id, ...pkg })
    })
  }

  getAllPackages(): SpiritualPackage[] {
    return Array.from(this.packages.values()).sort((a, b) => a.cityName.localeCompare(b.cityName))
  }

  getActivePackages(): SpiritualPackage[] {
    return this.getAllPackages().filter((p) => p.status === "active")
  }

  getInactivePackages(): SpiritualPackage[] {
    return this.getAllPackages().filter((p) => p.status === "inactive")
  }

  getPackageById(id: string): SpiritualPackage | undefined {
    return this.packages.get(id)
  }

  addPackage(cityName: string, dayPlans: DayPlan[]): SpiritualPackage {
    this.packageIdCounter++
    const id = `spiritual_pkg_${String(this.packageIdCounter).padStart(3, "0")}`
    const now = new Date().toISOString().split("T")[0]

    const totalLocations = dayPlans.reduce((sum, day) => sum + day.locations.length, 0)

    const newPackage: SpiritualPackage = {
      id,
      cityName: cityName.trim(),
      totalLocations,
      dayPlans,
      status: "active",
      createdAt: now,
      updatedAt: now,
    }

    this.packages.set(id, newPackage)
    return newPackage
  }

  updatePackage(id: string, cityName: string, dayPlans: DayPlan[]): SpiritualPackage | null {
    const pkg = this.packages.get(id)
    if (!pkg) return null

    const totalLocations = dayPlans.reduce((sum, day) => sum + day.locations.length, 0)

    const updatedPackage: SpiritualPackage = {
      ...pkg,
      cityName: cityName.trim(),
      totalLocations,
      dayPlans,
      updatedAt: new Date().toISOString().split("T")[0],
    }

    this.packages.set(id, updatedPackage)
    return updatedPackage
  }

  togglePackageStatus(id: string): SpiritualPackage | null {
    const pkg = this.packages.get(id)
    if (!pkg) return null

    const updatedPackage: SpiritualPackage = {
      ...pkg,
      status: pkg.status === "active" ? "inactive" : "active",
      updatedAt: new Date().toISOString().split("T")[0],
    }

    this.packages.set(id, updatedPackage)
    return updatedPackage
  }

  deletePackage(id: string): boolean {
    return this.packages.delete(id)
  }
}

export const spiritualPackageDataStore = new SpiritualPackageDataStore()
