import { cookies } from "next/headers"

const CSRF_TOKEN_NAME = "csrf_token"
const CSRF_HEADER_NAME = "x-csrf-token"

/**
 * Generate a CSRF token for security (server-side)
 */
export async function generateCsrfToken(): Promise<string> {
  "use server"
  const token = crypto.randomUUID()
  const cookieStore = await cookies()

  cookieStore.set(CSRF_TOKEN_NAME, token, {
    httpOnly: true,
    secure: process.env.NODE_ENV === "production",
    sameSite: "strict",
    maxAge: 60 * 60 * 24, // 24 hours
  })

  return token
}

/**
 * Verify CSRF token from request (server-side)
 */
export async function verifyCsrfToken(token: string): Promise<boolean> {
  "use server"
  const cookieStore = await cookies()
  const storedToken = cookieStore.get(CSRF_TOKEN_NAME)

  if (!storedToken || storedToken.value !== token) {
    return false
  }

  return true
}

/**
 * Get CSRF token from cookies (server-side)
 */
export async function getCsrfToken(): Promise<string | undefined> {
  "use server"
  const cookieStore = await cookies()
  return cookieStore.get(CSRF_TOKEN_NAME)?.value
}

export { CSRF_HEADER_NAME }
