interface RateLimitRecord {
  count: number
  resetTime: number
}

interface RateLimitStore {
  [key: string]: RateLimitRecord
}

const store: RateLimitStore = {}

let lastCleanup = Date.now()
const CLEANUP_INTERVAL = 60000 // 1 minute

function cleanupExpiredRecords(): void {
  const now = Date.now()
  if (now - lastCleanup < CLEANUP_INTERVAL) {
    return
  }

  lastCleanup = now
  for (const key of Object.keys(store)) {
    if (store[key] && now > store[key].resetTime) {
      delete store[key]
    }
  }
}

interface RateLimitOptions {
  interval: number // in milliseconds
  maxRequests: number
}

interface RateLimitResult {
  success: boolean
  remaining: number
  resetTime: number
}

/**
 * Simple in-memory rate limiting with cleanup
 * For production, use Redis or similar distributed store
 */
export function rateLimit(
  identifier: string,
  options: RateLimitOptions = { interval: 60000, maxRequests: 10 },
): RateLimitResult {
  // Run cleanup periodically
  cleanupExpiredRecords()

  const now = Date.now()
  const record = store[identifier]

  // Clean up expired record for this identifier
  if (record && now > record.resetTime) {
    delete store[identifier]
  }

  // Initialize or get current record
  const current = store[identifier] || {
    count: 0,
    resetTime: now + options.interval,
  }

  // Check if limit exceeded
  if (current.count >= options.maxRequests) {
    return {
      success: false,
      remaining: 0,
      resetTime: current.resetTime,
    }
  }

  // Increment counter
  current.count++
  store[identifier] = current

  return {
    success: true,
    remaining: options.maxRequests - current.count,
    resetTime: current.resetTime,
  }
}

/**
 * Client-side rate limit check using localStorage
 * This provides basic protection but should be paired with server-side validation
 */
export function clientRateLimit(
  key: string,
  maxAttempts = 5,
  windowMs = 60000,
): { allowed: boolean; remaining: number; resetIn: number } {
  if (typeof window === "undefined") {
    return { allowed: true, remaining: maxAttempts, resetIn: 0 }
  }

  const storageKey = `rate_limit_${key}`
  const now = Date.now()

  try {
    const stored = localStorage.getItem(storageKey)
    let record: { count: number; resetTime: number } = stored
      ? JSON.parse(stored)
      : { count: 0, resetTime: now + windowMs }

    // Reset if window has passed
    if (now > record.resetTime) {
      record = { count: 0, resetTime: now + windowMs }
    }

    if (record.count >= maxAttempts) {
      return {
        allowed: false,
        remaining: 0,
        resetIn: Math.max(0, record.resetTime - now),
      }
    }

    record.count++
    localStorage.setItem(storageKey, JSON.stringify(record))

    return {
      allowed: true,
      remaining: maxAttempts - record.count,
      resetIn: Math.max(0, record.resetTime - now),
    }
  } catch {
    // If localStorage fails, allow the request
    return { allowed: true, remaining: maxAttempts, resetIn: 0 }
  }
}

/**
 * Clear client-side rate limit (e.g., after successful login)
 */
export function clearClientRateLimit(key: string): void {
  if (typeof window === "undefined") return
  try {
    localStorage.removeItem(`rate_limit_${key}`)
  } catch {
    // Ignore errors
  }
}
