import { z } from "zod"

export const packageLocationSchema = z.object({
  id: z.string(),
  name: z.string().min(1, "Location name is required").max(200, "Name is too long"),
  image: z.string().min(1, "Image is required"),
  itineraryDescription: z
    .string()
    .min(10, "Description must be at least 10 characters")
    .max(1000, "Description is too long"),
  otherDetails: z.string().max(1000, "Other details are too long"),
})

export const packageVehicleSchema = z.object({
  id: z.string(),
  vehicleId: z.string().min(1, "Vehicle is required"),
  vehicleName: z.string(),
  vehiclePhoto: z.string(),
  perDayPrice: z.number().min(0, "Price must be positive"),
})

export const packagePricingSchema = z.object({
  b2cWeekdayPerHead: z.number().min(0, "Price must be positive"),
  b2cWeekendPerHead: z.number().min(0, "Price must be positive"),
  b2bAgentPricing: z.number().min(0, "Price must be positive"),
})

export const dayPlanSchema = z.object({
  dayNumber: z.number().min(1).max(15),
  locations: z.array(packageLocationSchema).min(1, "At least one location is required"),
  vehicles: z.array(packageVehicleSchema).min(1, "At least one vehicle is required"),
  pricing: packagePricingSchema,
})

export const customPackageFormSchema = z.object({
  cityName: z.string().min(1, "City name is required").max(100, "City name is too long"),
  category: z.enum(["adventure", "thrill", "suspense", "spiritual", "others"], {
    required_error: "Category is required",
  }),
  dayPlans: z.array(dayPlanSchema).min(1, "At least one day plan is required"),
})

export type PackageLocationInput = z.infer<typeof packageLocationSchema>
export type PackageVehicleInput = z.infer<typeof packageVehicleSchema>
export type PackagePricingInput = z.infer<typeof packagePricingSchema>
export type DayPlanInput = z.infer<typeof dayPlanSchema>
export type CustomPackageFormInput = z.infer<typeof customPackageFormSchema>
