import { z } from "zod"

export const packageVehicleSchema = z.object({
  id: z.string(),
  photo: z.string().min(1, "Vehicle photo is required"),
  vehicleName: z.string().min(2, "Vehicle name must be at least 2 characters").max(100, "Vehicle name is too long"),
  perDayPrice: z.number().min(100, "Per day price must be at least ₹100").max(100000, "Price is too high"),
  passengerLimit: z.number().min(1, "Passenger limit must be at least 1").max(50, "Passenger limit is too high"),
})

export const fixPackageSchema = z
  .object({
    name: z
      .string()
      .min(3, "Package name must be at least 3 characters")
      .max(200, "Package name is too long")
      .regex(/^[a-zA-Z0-9\s\-&()]+$/, "Package name contains invalid characters"),
    locationDetails: z
      .string()
      .min(10, "Location details must be at least 10 characters")
      .max(500, "Location details are too long"),
    itineraryDetails: z
      .string()
      .min(20, "Itinerary details must be at least 20 characters")
      .max(2000, "Itinerary details are too long"),
    otherDetails: z
      .string()
      .min(10, "Other details must be at least 10 characters")
      .max(2000, "Other details are too long"),
    originalPrice: z.number().min(1000, "Original price must be at least ₹1,000").max(1000000, "Price is too high"),
    discountedPrice: z.number().min(500, "Discounted price must be at least ₹500").max(1000000, "Price is too high"),
    totalDays: z.number().min(1, "Total days must be at least 1").max(365, "Duration cannot exceed 365 days"),
    vehicles: z.array(packageVehicleSchema).default([]),
  })
  .refine((data) => data.discountedPrice <= data.originalPrice, {
    message: "Discounted price must be less than or equal to original price",
    path: ["discountedPrice"],
  })

export type FixPackageInput = z.infer<typeof fixPackageSchema>
export type PackageVehicleInput = z.infer<typeof packageVehicleSchema>
