import { NextResponse } from "next/server"
import type { NextRequest } from "next/server"

const PUBLIC_ADMIN_PATHS = [
  "/admin/login",
  "/admin/forget-password",
  "/admin/email-otp-confirmation",
  "/admin/reset-password",
]

const PROTECTED_AGENT_PATHS = ["/agent"]
const PROTECTED_HOTEL_PATHS = ["/hotel"]

function addSecurityHeaders(response: NextResponse): NextResponse {
  response.headers.set("X-Frame-Options", "DENY")
  response.headers.set("X-XSS-Protection", "1; mode=block")
  response.headers.set("X-Content-Type-Options", "nosniff")
  response.headers.set("Referrer-Policy", "strict-origin-when-cross-origin")
  response.headers.set("Permissions-Policy", "camera=(), microphone=(), geolocation=()")
  return response
}

function isValidSession(sessionCookie: string | undefined, expectedRole?: string): boolean {
  if (!sessionCookie) {
    return false
  }

  try {
    const parsed = JSON.parse(decodeURIComponent(sessionCookie))
    if (!parsed.email || !parsed.role) {
      return false
    }
    if (expectedRole && parsed.role !== expectedRole) {
      return false
    }
    return true
  } catch {
    return false
  }
}

export function middleware(request: NextRequest) {
  const { pathname } = request.nextUrl
  const adminSessionCookie = request.cookies.get("admin_session")?.value
  const agentSessionCookie = request.cookies.get("agent_session")?.value
  const hotelSessionCookie = request.cookies.get("hotel_session")?.value

  if (pathname.startsWith("/admin")) {
    const isPublicAuthPath = PUBLIC_ADMIN_PATHS.some((path) => pathname === path)

    if (!isPublicAuthPath) {
      // All non-public admin paths require authentication
      if (!isValidSession(adminSessionCookie, "admin")) {
        const loginUrl = new URL("/login", request.url)
        loginUrl.searchParams.set("redirect", pathname)
        const response = NextResponse.redirect(loginUrl)
        return addSecurityHeaders(response)
      }
    } else if (isValidSession(adminSessionCookie, "admin")) {
      // Redirect authenticated users away from auth pages
      const response = NextResponse.redirect(new URL("/admin/dashboard", request.url))
      return addSecurityHeaders(response)
    }
  }

  if (pathname.startsWith("/agent")) {
    if (!isValidSession(agentSessionCookie, "agent")) {
      const loginUrl = new URL("/login", request.url)
      loginUrl.searchParams.set("redirect", pathname)
      const response = NextResponse.redirect(loginUrl)
      return addSecurityHeaders(response)
    }
  }

  if (pathname.startsWith("/hotel")) {
    if (!isValidSession(hotelSessionCookie, "hotel")) {
      const loginUrl = new URL("/login", request.url)
      loginUrl.searchParams.set("redirect", pathname)
      const response = NextResponse.redirect(loginUrl)
      return addSecurityHeaders(response)
    }
  }

  const response = NextResponse.next()
  return addSecurityHeaders(response)
}

export const config = {
  matcher: ["/((?!api|_next/static|_next/image|favicon.ico|.*\\.(?:svg|png|jpg|jpeg|gif|webp)$).*)"],
}
